// This file is a part of Framsticks SDK.  http://www.framsticks.com/
// Copyright (C) 1999-2020  Maciej Komosinski and Szymon Ulatowski.
// See LICENSE.txt for details.

#include "genoobj.h"
#include <frams/util/extvalue.h>
#include <frams/model/autoname.h>
#include "collectionobj.h"

#define FIELDSTRUCT GenoObj
ParamEntry geno_paramtab[] =
{
	{ "Geno", 1, 16, "Geno", "All information about a single genotype.\nThis is a genetics-only object which does not contain any performance data. See also: Genotype class" },
	{ "name", 0, PARAM_NOSTATIC, "Name", "s 0 40", GETSET(name), },
	{ "rawgenotype", 0, PARAM_NOSTATIC | PARAM_READONLY, "Raw genotype", "s 1", GETONLY(genotype), "Genotype, excluding the format specifier" },
	{ "info", 0, PARAM_NOSTATIC, "Info", "s 1", GETSET(info), "Additional information or comments", },
	{ "format", 0, PARAM_NOSTATIC | PARAM_READONLY, "Format", "s", GETONLY(format), "Genotype format", },
	{ "genotype", 0, PARAM_NOSTATIC | PARAM_READONLY, "Genotype", "s 1", GETONLY(string), "Genes as a string of characters", },
	{ "isValid", 0, PARAM_NOSTATIC | PARAM_READONLY | PARAM_DEPRECATED, "Valid", "d 0 1", GETONLY(isvalid), "Use 'is_valid' instead of 'isValid'." },
	{ "is_valid", 0, PARAM_NOSTATIC, "Validity", "d -1 1 -1", GETSET(isvalid),
	"0 = invalid genotype\n"
	"1 = valid genotype\n"
	"-1 = validity is not known. This is a transient state. The value of \"is_valid\" will never be -1 when read. It is safe to treat is_valid as boolean in statements like \"if (g.is_valid) ...\". Setting \"is_valid=-1\" will make it 0 or 1 again. This third state (-1) is only needed for loading Genotype objects from files where the \"is_valid\" field might not be present."
	},
	{ "getConverted", 0, PARAM_NOSTATIC, "Get converted genotype", "p oGeno(s format)", PROCEDURE(p_getconvert), },
	{ "getConvertedWithCheckpoints", 0, PARAM_NOSTATIC, "Get converted genotype", "p oGeno(s format)", PROCEDURE(p_getconvert_ch), "See also Model.newWithCheckpoints()" },
	{ "f0genotype", 0, PARAM_NOSTATIC | PARAM_READONLY, "f0 genotype", "s 1", GETONLY(f0genotype), "converted to f0 genotype", },
	{ "new", 0, 0, "create new empty object", "p oGeno()", PROCEDURE(p_new), },
	{ "newFromString", 0, 0, "create new object from supplied string argument", "p oGeno(s genotype)", PROCEDURE(p_newfromstring), },
	{ "newFrom", 0, 0, "create new object", "p oGeno(s genotype,s format,s name,s description)", PROCEDURE(p_newfrom), },
	{ "autoname", 0, PARAM_NOSTATIC | PARAM_READONLY, "Autogenerated name", "s", GETONLY(autoname), },
	{ "toVector", 0, PARAM_READONLY | PARAM_NOSTATIC, "serialization support", "oVector", GETONLY(toVector), },
	{ "newFromVector", 0, 0, "serialization support", "p oGeno(oVector)", PROCEDURE(p_newfromvector), },
	{ 0, 0, 0, },
};
#undef FIELDSTRUCT

void GenoObj::get_isvalid(ExtValue *ret)
{
	ret->setInt(isValid());
}

int GenoObj::set_isvalid(const ExtValue *v)
{
	paInt n = v->getInt();
	if (getValid() != n)
	{
		setValid(n);
		return PSET_CHANGED;
	}
	return 0;
}

void GenoObj::get_genotype(ExtValue *ret)
{
	ret->setString(getGenes());
}

void GenoObj::get_name(ExtValue *ret)
{
	ret->setString(getName());
}

void GenoObj::get_autoname(ExtValue *ret)
{
	Model m(*this, Model::SHAPETYPE_UNKNOWN);
	ret->setString(AutoName::makeName(m));
}

int GenoObj::set_name(const ExtValue *v)
{
	setName(v->getString());
	return PSET_CHANGED;
}

void GenoObj::get_info(ExtValue *ret)
{
	ret->setString(getComment());
}

void GenoObj::get_string(ExtValue *ret)
{
	ret->setString(getGenesAndFormat());
}

void GenoObj::get_format(ExtValue *ret)
{
	ret->setString(getFormat());
}

int GenoObj::set_info(const ExtValue *v)
{
	setComment(v->getString());
	return PSET_CHANGED;
}

void GenoObj::get_f0genotype(ExtValue *ret)
{
	ret->setString(getConverted(Geno::F0_FORMAT_LIST).getGenes());
}

SString GenoObj::formatFromExtValue(ExtValue& v)
{
	if (v.getType() == TInt)
	{
		char ch = v.getInt();
		return SString(&ch, 1);
	}
	if (v.getType() == TString)
	{
		return v.getString();
	}
	return Geno::FORMAT_INVALID;
}

void GenoObj::p_getconvert(ExtValue *args, ExtValue *ret)
{
	*ret = makeDynamicObjectAndDecRef(new Geno(getConverted(formatFromExtValue(args[0]))));
}

void GenoObj::p_getconvert_ch(ExtValue *args, ExtValue *ret)
{
	*ret = makeDynamicObjectAndDecRef(new Geno(getConverted(formatFromExtValue(args[0]), NULL, true)));
}

void GenoObj::p_new(ExtValue *args, ExtValue *ret)
{
	*ret = makeDynamicObjectAndDecRef(new Geno());
}

void GenoObj::p_newfromstring(ExtValue *args, ExtValue *ret)
{
	*ret = makeDynamicObjectAndDecRef(new Geno(args[0].getString()));
}

void GenoObj::p_newfrom(ExtValue *args, ExtValue *ret)
{
	*ret = makeDynamicObjectAndDecRef(new Geno(args[3].getString(), formatFromExtValue(args[2]),
		args[1].getString(), args[0].getString()));
}

Param& GenoObj::getStaticParam()
{
#ifdef __CODEGUARD__
	static GenoObj static_genoobj;
	static Param static_genoparam(geno_paramtab, &static_genoobj);
#else
	static Param static_genoparam(geno_paramtab);
#endif
	return static_genoparam;
}

Param& GenoObj::getDynamicParam()
{
	static Param dynamic_genoparam(geno_paramtab);
	return dynamic_genoparam;
}

ParamInterface* GenoObj::getInterface() { return &getStaticParam(); }

ExtObject GenoObj::makeStaticObject(Geno* g)
{
	return ExtObject(&getStaticParam(), (void*)g);
}

ExtObject GenoObj::makeDynamicObject(Geno* g)
{
	return ExtObject(&getDynamicParam(), (DestrBase*)g);
}

ExtObject GenoObj::makeDynamicObjectAndDecRef(Geno* g)
{
	const ExtObject& o = makeDynamicObject(g);
	g->decref();
	return o;
}

Geno* GenoObj::fromObject(const ExtValue& v, bool warn)
{
	return (Geno*)v.getObjectTarget(getStaticParam().getName(), warn);
}

void GenoObj::get_toVector(ExtValue *ret)
{
	VectorObject *vec = new VectorObject;
	vec->data += new ExtValue(getGenesAndFormat());
	vec->data += new ExtValue(getName());
	vec->data += new ExtValue(getComment());
	ret->setObject(ExtObject(&VectorObject::par, vec));
}

void GenoObj::p_newfromvector(ExtValue *args, ExtValue *ret)
{
	VectorObject *vec = VectorObject::fromObject(args->getObject());
	if (vec && (vec->data.size() >= 3))
	{
		SString g = vec->get(0) ? vec->get(0)->getString() : SString::empty();
		SString n = vec->get(1) ? vec->get(1)->getString() : SString::empty();
		SString c = vec->get(2) ? vec->get(2)->getString() : SString::empty();
		*ret = makeDynamicObjectAndDecRef(new Geno(g, Geno::FORMAT_UNKNOWN, n, c));
	}
	else
		ret->setEmpty();
}

/////////////

REGISTER_DESERIALIZABLE(GenoObj)
