package com.framsticks.params;

import com.framsticks.params.types.UniqueListParam;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.UnimplementedException;
import com.framsticks.util.UnsupportedOperationException;
import com.framsticks.util.lang.Casting;
import com.framsticks.util.lang.Numbers;
import org.apache.log4j.Logger;

import java.util.*;

/**
 * @author Piotr Sniegowski
 */
public class UniqueListAccess extends ListAccess {

	private static final Logger log = Logger.getLogger(UniqueListAccess.class);

	Map<String, Object> map;

	final String uidName;

	public UniqueListAccess(AccessInterface elementAccess, String uidName) {
		super(elementAccess);
		this.uidName = uidName;
	}

	public static Integer getUidNumber(String uid) {
		try {
			return Integer.valueOf(uid.substring(1));
		} catch (NumberFormatException e) {
			return null;
		}
	}

	public static class UidComparator implements Comparator<String> {

		protected String name;

		/**
		 * @param name
		 */
		public UidComparator(String name) {
			this.name = name;
		}

		@Override
		public int compare(String a, String b) {
			if (a.equals(b)) {
				return 0;
			}
			int diff = a.length() - b.length();
			if (diff != 0) {
				return diff;
			}
			Integer au = getUidNumber(a);
			Integer bu = getUidNumber(b);
			if (au == null || bu == null) {
				throw new FramsticksException().msg("comparator failure").arg("left", a).arg("right", b).arg("in", this);
			}
			return au - bu;
		}

		@Override
		public String toString() {
			return "comparator " + name;
		}


	}

	@Override
	public Map<String, Object> createAccessee() {
		return new TreeMap<String, Object>(new UidComparator(elementAccess.toString()));
	}

	@Override
	public CompositeParam getParam(int i) {
		if ((i < 0) ||  (i >= map.size())) {
			return null;
		}
		Iterator<Map.Entry<String, Object>> iterator = map.entrySet().iterator();
		while (i > 0 && iterator.hasNext()) {
			iterator.next();
			--i;
		}
		if (i > 0) {
			return null;
		}
		if (!iterator.hasNext()) {
			return null;
		}
		return paramBuilder.id(getUidOf(iterator.next().getValue())).finish(CompositeParam.class);
	}

	@Override
	public CompositeParam getParam(String id) {
		Integer i = Numbers.parse(id, Integer.class);
		if (i != null) {
			return getParam(i);
		}
		Integer uidNumber = getUidNumber(id);
		if (uidNumber == null) {
			return null;
		}
		if (!map.containsKey(id)) {
			return null;
		}
		return paramBuilder.id(id).finish(CompositeParam.class);
	}

	@Override
	public String getId() {
		return "l " + elementAccess.getId() + " " + uidName;
	}

	@Override
	public int getParamCount() {
		return map.size();
	}

	@Override
	public <T> T get(int i, Class<T> type) {
		Iterator<Map.Entry<String, Object>> iterator = map.entrySet().iterator();
		while (i > 0 && iterator.hasNext()) {
			iterator.next();
			--i;
		}
		if (i > 0) {
			return null;
		}
		if (!iterator.hasNext()) {
			return null;
		}
		return Casting.tryCast(type, iterator.next().getValue());
	}

	@Override
	public <T> T get(String id, Class<T> type) {
		Integer i = Numbers.parse(id, Integer.class);
		if (i != null) {
			return get(i, type);
		}
		Integer uidNumber = getUidNumber(id);
		if (uidNumber == null) {
			return null;
		}
		return Casting.tryCast(type, map.get(id));
	}

	@Override
	public <T> T get(ValueParam param, Class<T> type) {
		return get(param.getId(), type);
	}

	public String getUidOf(Object value) {
		Object tmp = elementAccess.getSelected();
		elementAccess.select(value);
		String uid = elementAccess.get(uidName, String.class);
		elementAccess.select(tmp);
		return uid;
	}

	protected int setByUid(Object object, String uid) {
		if (uid == null) {
			uid = getUidOf(object);
			if (uid == null) {
				log.error("failed to set - missing uid");
				return 0;
			}
		}
		if (object == null) {
			map.remove(uid);
		} else {
			map.put(uid, object);
		}
		return 0;
	}

	@Override
	public <T> int set(int i, T value) {
		throw new UnsupportedOperationException().msg("accesing unique list through index");
	}

	@Override
	public <T> int set(String id, T value) {
		Integer i = Numbers.parse(id, Integer.class);
		if (i != null) {
			return set(i, value);
		}
		if (value == null) {
			return setByUid(null, id);
		}
		String uid = getUidOf(value);
		if (uid != null && id != null) {
			if (!id.equals(uid)) {
				log.error("uid mismatch with set key");
				return 0;
			}
			setByUid(value, uid);
			return 0;
		}
		if (uid != null) {
			setByUid(value, uid);
			return 0;
		}
		if (id != null) {
			setByUid(value, id);
			return 0;
		}
		log.error("missing both uid and id - failed to set");
		return 0;
	}

	@Override
	public <T> int set(ValueParam param, T value) {
		return set(param.getId(), value);
	}

	@Override
	public void clearValues() {
		map.clear();
	}

	@SuppressWarnings("unchecked")
	@Override
	public UniqueListAccess select(Object object) {
		assert (object instanceof Map);
		map = (Map<String, Object>) object;
		return this;
	}

	@Override
	public Object getSelected() {
		return map;
	}

	@Override
	public UniqueListAccess cloneAccess() {
		return new UniqueListAccess(elementAccess.cloneAccess(), uidName);
	}

	public String computeIdentifierFor(Object selected) {
		String uid = getUidOf(selected);
		if (uid == null) {
			log.error("missing uid field");
			return null;
		}
		return uid;
	}

	@Override
	public Iterable<Param> getParams() {
		return new Iterable<Param>() {

			@Override
			public Iterator<Param> iterator() {
				return new Iterator<Param>() {

					protected Iterator<Map.Entry<String, Object>> internal = map.entrySet().iterator();

					@Override
					public boolean hasNext() {
						return internal.hasNext();
					}

					@Override
					public Param next() {
						return paramBuilder.id(internal.next().getKey()).finish();
					}

					@Override
					public void remove() {
						throw new UnimplementedException().msg("remove element from list").arg("list", UniqueListAccess.this);

					}
				};
			}
		};
	}

	@Override
	public int getCompositeParamCount() {
		return map.size();
	}

	@Override
	public CompositeParam getCompositeParam(int number) {
		return getParam(number);
	}

	@Override
	public ParamBuilder buildParam(ParamBuilder builder) {
		return builder.name(containedTypeName + " list").type(UniqueListParam.class).containedTypeName(containedTypeName).uid(uidName);
	}

}
