package com.framsticks.params;

import org.apache.log4j.Logger;

import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.util.DoubleMap;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.lang.Strings;

import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

/**
 * Author: Piotr Śniegowski
 */
@FramsClassAnnotation
public class Registry {
	private static final Logger log = Logger.getLogger(Registry.class.getName());

	protected final DoubleMap<String, Class<?>> javaClasses = new DoubleMap<>();
	protected final DoubleMap<String, FramsClass> framsClasses = new DoubleMap<>();
	protected final Map<Class<?>, FramsClass> javaToFramsAssociation = new IdentityHashMap<>();

	/**
	 *
	 */
	public Registry() {
		// registerAndBuild(Registry.class);
		// registerAndBuild(FramsClass.class);
		// registerAndBuild(Param.class);
	}

	public void registerReflectedClass(String name, String id, Class<?> javaClass) {
		javaClasses.put(id, name, javaClass);
	}

	public void associate(Class<?> javaClass, FramsClass framsClass) {
		javaToFramsAssociation.put(javaClass, framsClass);
	}

	public Registry registerAndBuild(Class<?> javaClass) {
		register(javaClass);
		associate(javaClass, putFramsClass(FramsClass.build().forClass(javaClass)));
		for (Class<?> r : javaClass.getAnnotation(FramsClassAnnotation.class).register()) {
			registerAndBuild(r);
		}
		return this;
	}

	public FramsClass registerReflectedIfNeeded(Class<?> javaClass) {
		if (!javaToFramsAssociation.containsKey(javaClass)) {
			registerAndBuild(javaClass);
		}
		return javaToFramsAssociation.get(javaClass);
	}

	public Registry register(Class<?> javaClass) {
		FramsClassAnnotation a = javaClass.getAnnotation(FramsClassAnnotation.class);
		if (a == null) {
			throw new FramsticksException().msg("class is not annotated").arg("class", javaClass);
		}

		registerReflectedClass(FramsClassBuilder.getName(a, javaClass), FramsClassBuilder.getId(a, javaClass), javaClass);
		return this;
	}

	public @Nonnull ReflectionAccess createAccess(Class<?> javaClass) throws ConstructionException {
		try {
			if (!javaClasses.containsValue(javaClass)) {
				throw new FramsticksException().msg("java class is not registered");
			}
			if (!javaToFramsAssociation.containsKey(javaClass)) {
				throw new FramsticksException().msg("java class is not associated with any frams class");
			}
			return new ReflectionAccess(javaClass, javaToFramsAssociation.get(javaClass));
		}
		catch (FramsticksException e) {
			throw new FramsticksException().msg("failed to create access for java class").arg("class", javaClass).cause(e);
		}
	}

	public @Nonnull AccessInterface createAccess(String name, FramsClass framsClass) throws ConstructionException {
		// assert framsClasses.containsValue(framsClass);
		if (javaClasses.containsKey(name)) {
			return new ReflectionAccess(javaClasses.get(name), framsClass);
		}
		return new PropertiesAccess(framsClass);
	}

	public FramsClass putFramsClass(FramsClass framsClass) {
		log.debug("caching info for " + framsClass);
		framsClasses.put(framsClass.getId(), framsClass.getName(), framsClass);
		return framsClass;
	}

	public FramsClass getFramsClass(@Nonnull String identifier) {
		return framsClasses.get(identifier);
	}

	public static @Nonnull AccessInterface wrapAccessWithListIfNeeded(@Nonnull CompositeParam param, @Nonnull AccessInterface access) {
		return param.prepareAccessInterface(access);
	}

	public @Nonnull AccessInterface prepareAccess(CompositeParam param) throws ConstructionException {
		return wrapAccessWithListIfNeeded(param, createAccess(param.getContainedTypeName()));
	}

	public @Nonnull AccessInterface createAccess(@Nonnull String name) throws ConstructionException {
		try {
			Strings.assureNotEmpty(name);
			FramsClass framsClass = getFramsClass(name);
			if (framsClass == null) {
				throw new ConstructionException().msg("framsclass is missing");
			}

			return createAccess(name, framsClass);
		}
		catch (FramsticksException e) {
			throw new FramsticksException().msg("failed to create access for name").arg("name", name).cause(e);
		}
	}

	public FramsClass getFramsClassForJavaClass(Class<?> javaClass) {
		return javaToFramsAssociation.get(javaClass);
	}

	public Set<Class<?>> getReflectedClasses() {
		return javaClasses.getValues();
	}

	public Set<FramsClass> getFramsClasses() {
		return framsClasses.getValues();
	}

	@ParamAnnotation
	public Map<String, FramsClass> getFramsClassesById() {
		return framsClasses.getValuesById();
	}

	public void takeAllFrom(Registry source) {
		for (Class<?> javaClass : source.getReflectedClasses()) {
			register(javaClass);
		}
		for (FramsClass framsClass : source.getFramsClasses()) {
			putFramsClass(framsClass);
		}

	}

}
