package com.framsticks.params;


import javax.annotation.concurrent.Immutable;

import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;

/**
 * Based on c++ struct ParamEntry located in cpp/gdk/param.h
 * Here  it is a root of Param hierarchy.
 *
 * @author Jarek Szymczak <name.surname@gmail.com>, Mateusz Jarus
 * (please replace name and surname with my personal data)
 *
 * @author Piotr Śniegowski
 */
@Immutable
@FramsClassAnnotation(id = "prop", name = "prop", order = {"id", "name", "type", "flags", "help", "group", "extra"})
public abstract class Param {

	/** The parameter id. */
	protected final String id;

	/** The parameter name. */
	protected final String name;

	/** The help (description) concerning parameter. */
	protected final String help;

	/** The number of group, that parameter belongs to. */
	protected final Integer group;

	/** The getFlags stored as a bit sum. */
	protected final int flags;

	//TODO
	/** The variable determining whether the parameter is an extra parameter. */
	protected final int extra;

	public Param(ParamBuilder builder) {
		id = builder.getId();
		name = builder.getName();
		help = builder.getHelp();
		group = builder.getGroup();
		flags = builder.getFlags();
		extra = builder.getExtra();
	}

	@ParamAnnotation
	public String getId() {
		return id;
	}

	@ParamAnnotation
	public String getName() {
		return name;
	}

	@ParamAnnotation
	public String getHelp() {
		return help;
	}

	@ParamAnnotation
	public Integer getGroup() {
		return group;
	}

	@ParamAnnotation(def = "0")
	public Integer getFlags() {
		return flags;
	}

	@ParamAnnotation(id = "type")
	public abstract String getFramsTypeName();

	@ParamAnnotation
	public Integer getExtra() {
		return extra;
	}

	@Override
	public String toString() {
		return getId() + ":" + this.getClass().getSimpleName();
	}

	public boolean isNumeric() {
		return false;
	}

	public abstract Class<?> getStorageType();

	public boolean hasFlag(int flag) {
		return (flags & flag) != 0;
	}

	public boolean isUserHidden() {
		return (flags & ParamFlags.USERHIDDEN) != 0;
	}

	public static ParamBuilder build() {
		return new ParamBuilder();
	}

	public static ParamBuilder build(FramsClassBuilder builder) {
		return new ParamBuilder(builder);
	}

}
