package com.framsticks.gui.tree;

import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import javax.swing.ImageIcon;
import javax.swing.tree.TreePath;

import org.apache.log4j.Logger;

import com.framsticks.core.ListChange;
import com.framsticks.core.Node;
import com.framsticks.core.Path;
import com.framsticks.core.Tree;
import com.framsticks.gui.Frame;
import com.framsticks.gui.ImageProvider;
import com.framsticks.gui.TreeAtFrame;
import com.framsticks.params.AccessInterface;
import com.framsticks.params.CompositeParam;
import com.framsticks.params.EventListener;
import com.framsticks.params.FramsClass;
import com.framsticks.params.PrimitiveParam;
import com.framsticks.params.ValueParam;
import com.framsticks.params.types.EventParam;
import com.framsticks.params.types.ObjectParam;
import com.framsticks.params.types.StringParam;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.FutureHandler;
import com.framsticks.util.lang.Casting;
import com.framsticks.util.lang.Containers;
import com.framsticks.util.lang.Pair;
import com.framsticks.util.swing.TooltipConstructor;
import static com.framsticks.core.TreeOperations.*;

public class TreeNode extends AbstractNode {
	private static final Logger log = Logger.getLogger(TreeNode.class);


	protected static final AtomicInteger counter = new AtomicInteger();

	protected final WeakReference<Object> reference;
	protected final CompositeParam param;
	protected final TreeAtFrame treeAtFrame;
	protected final List<Pair<WeakReference<Object>, WeakReference<TreeNode>>> children = new LinkedList<>();
	protected final int number;
	protected final String textualPath;
	protected final ImageIcon imageIcon;
	protected final TreeModel treeModel;

	protected final List<EventListener<?>> listeners = new LinkedList<>();

	public TreeNode(TreeAtFrame treeAtFrame, Path path) {
		path.assureResolved();
		this.textualPath = path.getTextual();
		this.param = path.getTop().getParam();
		this.treeAtFrame = treeAtFrame;
		this.treeModel = treeAtFrame.getFrame().getTreeModel();
		this.imageIcon = ImageProvider.loadImage(TreeCellRenderer.findIconName(param));

		reference = new WeakReference<Object>(path.getTop().getObject());
		number = counter.getAndIncrement();

		/** Iterate over all EventParams and for matching ValueParams register listeners. */
		if (param instanceof ObjectParam) {
			AccessInterface access = bindAccess(path);
			FramsClass framsClass = access.getFramsClass();
			for (EventParam eventParam : Containers.filterInstanceof(framsClass.getParamEntries(), EventParam.class)) {
				if (!eventParam.getId().endsWith("_changed")) {
					continue;
				}
				String valueId = eventParam.getId().substring(0, eventParam.getId().length() - 8);
				final ValueParam valueParam = Casting.tryCast(ValueParam.class, framsClass.getParam(valueId));
				if (valueParam == null) {
					continue;
				}
				registerForEventParam(path, eventParam, valueParam);
			}
		}
	}

	protected <A> void tryAddListener(final Path path, final EventParam eventParam, Class<A> argumentType, final EventListener<A> listener) {
		treeAtFrame.getTree().addListener(path, eventParam, listener, argumentType, new FutureHandler<Void>(treeAtFrame.getFrame()) {
			@Override
			protected void result(Void result) {
				assert treeAtFrame.getFrame().isActive();
				log.debug("registered gui listener for " + eventParam + " at " + path);
				listeners.add(listener);
			}
		});
	}

	protected void registerForEventParam(Path path, EventParam eventParam, ValueParam valueParam) {
		/** TODO make this listener not bind hold the reference to this TreeNode, maybe hold WeakReference internally */
		if (valueParam instanceof PrimitiveParam) {

			tryAddListener(path, eventParam, Object.class, new EventListener<Object>() {
				@Override
				public void action(Object argument) {
					treeModel.loadPath(assurePath(), true);
				}
			});

		} else if (valueParam instanceof CompositeParam) {

			final CompositeParam compositeParam = (CompositeParam) valueParam;

			tryAddListener(path, eventParam, ListChange.class, new EventListener<ListChange>() {
				@Override
				public void action(ListChange listChange) {
					assert treeAtFrame.getTree().isActive();

					final Path listPath = assurePath().appendParam(compositeParam).tryFindResolution().assureResolved();
					log.debug("reacting to change " + listChange + " in " + listPath);

					if ((listChange.getAction().equals(ListChange.Action.Modify)) && (listChange.getPosition() == -1)) {
						// get(listPath, future);
						return;
					}
					final String id = listChange.getBestIdentifier();

					AccessInterface access = bindAccess(listPath);
					switch (listChange.getAction()) {
						case Add: {
							tryGet(listPath.getTree(), Path.appendString(listPath.getTextual(), id), new FutureHandler<Path>(treeAtFrame.getFrame()) {
								@Override
								protected void result(Path result) {
									final Frame frame = treeAtFrame.getFrame();
									assert frame.isActive();
									final TreePath treePath = frame.getTreeModel().convertToTreePath(listPath);
									treeModel.nodeStructureChanged(treePath);
									frame.updatePanelIfIsLeadSelection(treePath, listPath);
									log.debug("added " + id + "(" + result + ") updated " + treePath);
								}
							});
							break;
						}
						case Remove: {
							access.set(id, null);
							Frame frame = treeAtFrame.getFrame();
							treeModel.nodeStructureChanged(frame.getTreeModel().convertToTreePath(listPath));
							break;
						}
						case Modify: {
							tryGet(listPath.getTree(), Path.appendString(listPath.getTextual(), id), new FutureHandler<Path>(treeAtFrame.getFrame()) {
								@Override
								protected void result(Path result) {
									final Frame frame = treeAtFrame.getFrame();
									assert frame.isActive();
									final TreePath treePath = frame.getTreeModel().convertToTreePath(result);
									treeModel.nodeStructureChanged(treePath);
									frame.updatePanelIfIsLeadSelection(treePath, listPath);
								}
							});
							break;
						}
					}
				}
			});
		}

	}

	protected Path assurePath() {
		return Path.to(treeAtFrame.getTree(), textualPath).assureResolved();
	}

	public Node tryCreateNode() {
		Object child = lock();
		if (child == null) {
			return null;
		}
		return Path.to(treeAtFrame.getTree(), textualPath).assureResolved().getTop();
	}

	@Override
	public int getChildCount() {
		Object referent = lock();
		if (referent == null) {
			return 0;
		}
		AccessInterface access = bindAccessFor(referent);
		final int count = access.getCompositeParamCount();
		return count;
	}

	public TreeNode getTreeNodeForChild(Object child) {
		Iterator<Pair<WeakReference<Object>, WeakReference<TreeNode>>> i = children.iterator();
		while (i.hasNext()) {
			Pair<WeakReference<Object>, WeakReference<TreeNode>> p = i.next();
			Object object = p.first.get();
			if (object == null) {
				i.remove();
				continue;
			}
			TreeNode treeNode = p.second.get();
			if (treeNode == null) {
				i.remove();
				continue;
			}
			if (object == child) {
				return treeNode;
			}
		}
		return null;

		// WeakReference<GuiTreeNode> resultReference = children.get(child);
		// if (resultReference == null) {
		//	return null;
		// }
		// return resultReference.get();
	}

	@Override
	public AbstractNode getChild(int number) {
		Object referent = lock();
		if (referent == null) {
			throw new FramsticksException().msg("invalid state - missing referent");
		}
		AccessInterface access = bindAccessFor(referent);

		final int count = access.getCompositeParamCount();
		if (number >= count) {
			throw new FramsticksException().msg("invalid state - no child");
		}

		CompositeParam childParam = access.getCompositeParam(number);
		Object child = access.get(childParam, Object.class);
		if (child == null) {
			log.debug("returning dummy node for " + childParam + " in " + referent);
			return new EmptyNode(childParam);
		}

		TreeNode result = getTreeNodeForChild(child);
		if (result != null) {
			return result;
		}
		Path path = Path.to(treeAtFrame.getTree(), Path.appendString(textualPath, childParam.getId())).assureResolved();
		result = new TreeNode(treeAtFrame, path);

		children.add(Pair.make(new WeakReference<Object>(child), new WeakReference<TreeNode>(result)));

		return result;

	}

	public Object lock() {
		return reference.get();
	}

	@Override
	public int getIndexOfChild(AbstractNode child) {
		final TreeNode treeChild = Casting.tryCast(TreeNode.class, child);
		if (treeChild == null) {
			return -1;
		}
		final Object childObject = treeChild.lock();
		final Object parentObject = lock();
		if (childObject == null || parentObject == null) {
			return -1;
		}
		final AccessInterface access = bindAccessFor(parentObject);

		final int count = access.getCompositeParamCount();
		for (int i = 0; i < count; ++i) {
			Object c = access.get(access.getCompositeParam(i), Object.class);
			if (c == childObject) {
				return i;
			}
		}
		log.debug(child + " not found in " + this);
		return -1;
	}

	/**
	 * @return the param
	 */
	public CompositeParam getParam() {
		return param;
	}

	/**
	 * @return the tree
	 */
	public Tree getTree() {
		return treeAtFrame.getTree();
	}


	@Override
	public String toString() {
		return param.toString();
	}

	public static Node tryGetNode(TreePath treePath) {
		return Casting.throwCast(TreeNode.class, treePath.getLastPathComponent()).tryCreateNode();
	}

	@Override
	public boolean isLeaf() {
		Object referent = lock();
		if (referent == null) {
			return true;
		}
		return bindAccessFor(referent).getCompositeParamCount() == 0;
	}

	protected AccessInterface bindAccessFor(Object child) {
		return treeAtFrame.getTree().prepareAccess(param).select(child);
	}

	@Override
	public void render(TreeCellRenderer renderer) {
		String name = param.getId();

		Object child = lock();
		if (child != null) {
			AccessInterface access = bindAccessFor(child);

			StringParam nameParam = Casting.tryCast(StringParam.class, access.getParam("name"));

			if (nameParam != null) {
				name = access.get(nameParam, String.class);
			}

			renderer.setToolTipText(new TooltipConstructor()
				.append("frams", access.getId())
				.append("java", child.getClass().getCanonicalName())
				.append("access", access.getClass().getSimpleName())
				.append("name", name)
				.append("id", param.getId())
				.append("object", Integer.toHexString(System.identityHashCode(child)))
				.append("number", number)
				.append("textual path", textualPath)
				.build());
		} else {
			renderer.setToolTipText(new TooltipConstructor()
				.append("param", param)
				.append("textual path", textualPath)
				.build());
		}
		renderer.setText(name);
		renderer.setIcon(imageIcon);

	}

}
