package com.framsticks.core;


import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;

import org.apache.log4j.Logger;

import com.framsticks.communication.File;
import com.framsticks.params.AccessInterface;
import com.framsticks.params.EventListener;
import com.framsticks.params.FramsClass;
import com.framsticks.params.ListAccess;
import com.framsticks.params.Param;
import com.framsticks.params.PrimitiveParam;
import com.framsticks.params.UniqueListAccess;
import com.framsticks.params.Util;
import com.framsticks.params.types.EventParam;
import com.framsticks.params.types.ObjectParam;
import com.framsticks.params.types.ProcedureParam;
import com.framsticks.parsers.Loaders;
import com.framsticks.parsers.MultiParamLoader;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.Future;
import com.framsticks.util.dispatching.FutureHandler;
import com.framsticks.util.dispatching.RunAt;

import static com.framsticks.util.dispatching.Dispatching.*;

public final class TreeOperations {

	private static final Logger log = Logger.getLogger(TreeOperations.class);

	private TreeOperations() {
	}

	public static @Nonnull FramsClass processFetchedInfo(Tree tree, File file) {
		assert tree.isActive();
		FramsClass framsClass = Loaders.loadFramsClass(file.getContent());
		log.debug("process fetched info for " + tree + ": " + framsClass);
		tree.putInfoIntoCache(framsClass);
		return framsClass;
	}

	public static void processFetchedValues(Path path, List<File> files) {
		Tree tree = path.getTree();
		assert tree.isActive();
		assert files.size() == 1;
		assert path.isTheSame(files.get(0).getPath());

		if (!path.isResolved()) {
			AccessInterface access = tree.prepareAccess(path.getTop().getParam());
			Object child = access.createAccessee();
			assert child != null;
			if (path.size() == 1) {
				tree.assignRootObject(child);
			} else {
				bindAccess(path.getUnder()).set(path.getTop().getParam(), child);
			}
			path = path.appendResolution(child);
		}

		log.debug("process fetched values: " + path);
		Node node = path.getTop();
		MultiParamLoader loader = new MultiParamLoader();
		loader.setNewSource(files.get(0).getContent());
		loader.addBreakCondition(MultiParamLoader.Status.AfterObject);

		try {
			if (node.getParam() instanceof ObjectParam) {
				loader.addAccessInterface(bindAccess(node));
				loader.go();
				return;
			}

			ListAccess listAccess = (ListAccess) bindAccess(node);

			Set<String> oldValuesIds = new HashSet<>();
			for (Param p : listAccess.getParams()) {
				oldValuesIds.add(p.getId());
			}

			// listAccess.clearValues();

			AccessInterface elementAccess = listAccess.getElementAccess();
			AccessInterface clonerInterface = elementAccess.cloneAccess();

			loader.addAccessInterface(elementAccess);
			MultiParamLoader.Status status;
			int number = 0;
			while ((status = loader.go()) != MultiParamLoader.Status.Finished) {
				if (status == MultiParamLoader.Status.AfterObject) {
					AccessInterface accessInterface = loader.getLastAccessInterface();

					String id;
					if (listAccess instanceof UniqueListAccess) {
						id = ((UniqueListAccess) listAccess).computeIdentifierFor(accessInterface.getSelected());
					} else {
						id = Integer.toString(number);
					}
					++number;

					Object childTo = listAccess.get(id, Object.class);
					boolean newOne;
					if (childTo == null) {
						childTo = clonerInterface.createAccessee();
						newOne = true;
					} else {
						assert oldValuesIds.contains(id);
						newOne = false;
					}
					oldValuesIds.remove(id);
					clonerInterface.select(childTo);
					Util.takeAllNonNullValues(clonerInterface, accessInterface);
					if (newOne) {
						listAccess.set(id, childTo);
					}

					// listAccess.set(id, accessInterface.getSelected());
					accessInterface.select(null);

				}
			}
			/** It looks tricky for ArrayListAccess but should also work.
 			 *
			 * They should be sorted.
			 */
			for (String id : oldValuesIds) {
				listAccess.set(id, null);
			}

		} catch (FramsticksException e) {
			throw new FramsticksException().msg("exception occurred while loading").cause(e);
		}
	}

	public static FramsClass getInfo(Path path) {
		Tree tree = path.getTree();
		assert tree.isActive();
		log.debug("get info for: " + path);
		final String name = path.getTop().getParam().getContainedTypeName();
		return tree.getInfoFromCache(name);
	}

	public static void findInfo(final Path path, final Future<FramsClass> future) {
		log.debug("find info for: " + path);
		try {
			Tree tree = path.getTree();
			assert tree.isActive();
			final FramsClass framsClass = getInfo(path);
			if (framsClass != null) {
				future.pass(framsClass);
				return;
			}
			tree.info(path, future);
		} catch (FramsticksException e) {
			future.handle(e);
		}
	}



	public static @Nonnull AccessInterface bindAccess(Tree tree, String path) {
		log.debug("bind access for textual: " + path + " in " + tree);
		return bindAccess(Path.to(tree, path));
	}

	public static @Nonnull AccessInterface bindAccess(Node node) {
		Tree tree = node.getTree();
		assert tree.isActive();
		assert node.getObject() != null;

		try {
			return tree.prepareAccess(node.getParam()).select(node.getObject());
		} catch (FramsticksException e) {
			throw new FramsticksException().msg("failed to prepare access for param").arg("param", node.getParam()).cause(e);
			// log.error("failed to bind access for " + node.getParam() + ": " + e);
		}
	}

	public static @Nonnull AccessInterface bindAccess(Path path) {
		assert path.getTree().isActive();
		path.assureResolved();
		log.debug("bind access for: " + path);
		return bindAccess(path.getTop());
	}

	public static void set(final Path path, final PrimitiveParam<?> param, final Object value, final Future<Integer> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				tree.set(path, param, value, future);
			}
		});
	}

	public static void call(final Path path, final ProcedureParam param, final Object[] arguments, final Future<Object> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				tree.call(path, param, arguments, future);
			}
		});
	}

	public static <A> void addListener(final Path path, final EventParam param, final EventListener<A> listener, final Class<A> argument, final Future<Void> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				tree.addListener(path, param, listener, argument, future);
			}
		});
	}

	public static void removeListener(final Path path, final EventParam param, final EventListener<?> listener, final Future<Void> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				tree.removeListener(path, param, listener, future);
			}
		});
	}


	/**
	 *
	 * If StackOverflow occurs in that loop in LocalTree it is probably caused
	 * the by the fact, that get operation may find the object, but Path resolution
	 * cannot.
	 * */
	public static void tryGet(final Tree tree, final String targetPath, final Future<Path> future) {
		log.debug("resolve textual: " + targetPath + " for " + tree);
		dispatchIfNotActive(tree, new RunAt<Tree>(future) {

			@Override
			protected void runAt() {
				final Path path = Path.tryTo(tree, targetPath).tryResolveIfNeeded();
				log.debug("found: " + path);
				if (path.isResolved()) {
					future.pass(path);
					return;
				}

				tree.get(path, new FutureHandler<Path>(future) {
					@Override
					protected void result(Path result) {
						// if (result.isResolved(targetPath)) {
						// 	future.pass(result);
						// 	return;
						// }
						log.debug("retrying resolve textual: " + targetPath + " for " + tree + " with " + result);
						tryGet(tree, targetPath, future);
					}
				});
			}
		});
	}

	public static FramsClass getInfoFromCache(Path path) {
		assert path.getTree().isActive();
		return path.getTree().getInfoFromCache(path.getTop().getParam().getContainedTypeName());
	}

}
