// This file is a part of Framsticks GDK library.
// Copyright (C) 2002-2006  Szymon Ulatowski.  See LICENSE.txt for details.
// Refer to http://www.frams.alife.pl/ for further information.

#include "neuroimpl.h"
#include "neurofactory.h"
#ifndef NOCREATUREOBJECT
#include "creature.h"
#include "creatmechobj.h"
#include "livegroups.h"
#include "simul.h"
#endif

const int NeuroImpl::ENDDRAWING=-9999;
const int NeuroImpl::MAXDRAWINGXY=0xffff;

int NeuroNetImpl::mytags_id=0;

/////////////////////////////////////////////////////////

#define FIELDSTRUCT NeuroNetConfig
static ParamEntry nncfg_paramtab[]=
{
{"Creature: Neurons",1,2,"nnsim",},
{"randinit",1,0,"Random initialization","f 0 10 0.01",FIELD(randominit),"Initialize all neuron states with uniform distribution random numbers"},
{"touchrange",1,0,"T receptor range","f 0 100 1",FIELD(touchrange),},
{0,0,0,},
}; 
#undef FIELDSTRUCT

NeuroNetConfig::NeuroNetConfig()
	:par(nncfg_paramtab,this),
	 randominit(0.01),
	 touchrange(1)
{}

NeuroNetConfig NeuroNetConfig::globalconfig;

/////////////////////////////////////////////////////////////////

NeuroNetImpl::NeuroNetImpl(Model& model, NeuroNetConfig& conf
#ifdef NEURO_SIGNALS
, ChannelSpace *ch
#endif
)
	:mod(model),config(conf),
	 isbuilt(1),errorcount(0)
#ifdef NEURO_SIGNALS
,channels(ch)
#endif
{
if (!mytags_id) mytags_id=mod.userdata.newID();

Neuro *n;
NeuroImpl *ni;
Joint *j;
int i;
DB(printf("makeNeuroNet(%p)\n",&mod));

minorder=3; maxorder=0;
errorcount=0;

for (i=0;j=mod.getJoint(i);i++)
	j->flags&=~(4+8); // todo: !!!neuroitems shouldn't use model fields!!!

for (i=0;n=mod.getNeuro(i);i++)
	{
	ni=NeuroFactory::createNeuroImpl(n);
	n->userdata[mytags_id]=ni;
	if (!ni) { errorcount++;
		FMprintf("NeuroNetImpl","create",FMLV_WARN,"neuron #%d (%s) implementation not available",
			 i,(const char*)n->getClassName());
		continue; } // implementation not available?!
	ni->owner=this;
	ni->neuro=n;
	ni->readParam();
	}

for (i=0;n=mod.getNeuro(i);i++)
	{
	n->state+=(rnd01-0.5)*config.randominit;
	ni=(NeuroImpl*)n->userdata[mytags_id];
	if (!ni) continue;
	if (!ni->lateinit())
		{ ni->status=NeuroImpl::InitError; errorcount++;
		FMprintf("NeuroNetImpl","create",FMLV_WARN,"neuron #%d (%s) initialization failed",
			 i,(const char*)n->getClassName());
		continue; }
	ni->status=NeuroImpl::InitOk;
	int order=ni->getSimOrder();
	if (order<0) order=0; else if (order>2) order=2;
	if (order<minorder) minorder=order;
	if (order>maxorder) maxorder=order;
	neurons[order]+=ni;
	if (ni->getNeedPhysics())
		neurons[3]+=ni;
	}
cnode=mod.delmodel_list.add(STATRICKCALLBACK(this,&NeuroNetImpl::destroyNN,0));
}

void NeuroNetImpl::destroyNN(CALLBACKARGS)
{
if (!isbuilt) return;
DB(printf("destroyNeuroNet(%p)\n",&mod));
NeuroImpl *ni;
Neuro *n;
for (int i=0;n=mod.getNeuro(i);i++)
	{
	ni=(NeuroImpl*)n->userdata[mytags_id];
	delete ni;
	n->userdata[mytags_id]=0;
	}
mod.delmodel_list.remove(cnode);
isbuilt=0; errorcount=0;
delete this;
}

NeuroNetImpl::~NeuroNetImpl()
{
destroyNN(0,0);
}

void NeuroNetImpl::simulateNeuroNet()
{
NeuroImpl *ni;
for (int order=minorder;order<=maxorder;order++)
	{
	int i;
	SList &nlist=neurons[order];
	for (i=0;ni=(NeuroImpl*)nlist(i);i++)
		ni->go();
	for (i=0;ni=(NeuroImpl*)nlist(i);i++)
		ni->commit();
	}
}

void NeuroNetImpl::simulateNeuroPhysics()
{
NeuroImpl *ni;
int i;
SList &nlist=neurons[3];
for (i=0;ni=(NeuroImpl*)nlist(i);i++)
	ni->goPhysics();
}

///////////////////////////////////////////////

void NeuroImpl::setChannelCount(int c)
{
if (c<1) c=1;
if (c==channels) return;
if (c<channels) {channels=c; chstate.trim(c-1); chnewstate.trim(c-1); return;}
double s=getState(channels-1);
chnewstate.setSize(c-1);
chstate.setSize(c-1);
for(int i=channels;i<c;i++)
	{chstate(i-1)=s; chnewstate(i-1)=s;}
channels=c;
}

void NeuroImpl::setState(double st,int channel)
{
validateNeuroState(st);
if (channel>=channels) channel=channels-1;
if (channel<=0) {newstate=st;return;}
chnewstate(channel-1)=st;
}

void NeuroImpl::setCurrentState(double st,int channel)
{
validateNeuroState(st);
if (channel>=channels) channel=channels-1;
if (channel<=0) {neuro->state=st; return;}
chstate(channel-1)=st;
}

double NeuroImpl::getNewState(int channel)
{
if (neuro->flags&Neuro::HoldState) return getState(channel);
if (channel>=channels) channel=channels-1;
if (channel<=0) {return newstate;}
return chnewstate(channel-1);
}

double NeuroImpl::getState(int channel)
{
if (channel>=channels) channel=channels-1;
if (channel<=0) return neuro->state;
return chstate(channel-1);
}

void NeuroImpl::commit()
{
if (!(neuro->flags&Neuro::HoldState))
	{
	if (channels>1)
		chstate=chnewstate;
	neuro->state=newstate;
	}
}

int NeuroImpl::getInputChannelCount(int i)
{
if ((i<0)||(i >= neuro->getInputCount())) return 1;
Neuro *nu=neuro->getInput(i);
NeuroImpl *ni=NeuroNetImpl::getImpl(nu);
if (!ni) return 1;
return ni->channels;
}

double NeuroImpl::getInputState(int i,int channel)
{
if ((i<0)||(i >= neuro->getInputCount())) return 0;
Neuro *nu=neuro->getInput(i);
if (channel<=0) return nu->state;
NeuroImpl *ni=NeuroNetImpl::getImpl(nu);
if (!ni) return nu->state;
if (channel>=ni->channels) channel=ni->channels-1;
if (!channel) return nu->state;
return ni->chstate(channel-1);
}

double NeuroImpl::getWeightedInputState(int i, int channel)
{
if ((i<0)||(i >= neuro->getInputCount())) return 0;
float w;
Neuro *nu=neuro->getInput(i,w);
if (channel<=0) return nu->state * w;
NeuroImpl *ni=NeuroNetImpl::getImpl(nu);
if (!ni) return nu->state * w;
if (channel>=ni->channels) channel=ni->channels-1;
if (!channel) return w * nu->state;
return w * ni->chstate(channel-1);
}

double NeuroImpl::getInputSum(int startwith)
{
if (startwith<0) return 0;
Neuro *inp;
double sum=0.0;
while(inp=neuro->getInput(startwith++))
	sum+=inp->state;
return sum;
}

double NeuroImpl::getWeightedInputSum(int startwith)
{
if (startwith<0) return 0;
Neuro *inp;
double sum=0.0;
float w;
while(inp=neuro->getInput(startwith++,w))
	sum+=inp->state*w;
return sum;
}

void NeuroImpl::readParam()
{
static Param par;
if (!paramentries) return;
par.setParamTab(paramentries);
par.select(this);
par.setDefault();
int zero=0;
par.load2(neuro->getClassParams(),zero);
}

/////////////////////////////

#ifdef NEURO_SIGNALS
#define NEUROIMPL_SIGNAL_PROPS 1
#else
#define NEUROIMPL_SIGNAL_PROPS 0
#endif

#define FIELDSTRUCT NeuroImpl
ParamEntry neuroimpl_tab[]=
{
{"Neuro",1,26+NEUROIMPL_SIGNAL_PROPS,"Neuro","Live Neuron object."},

{"getInputState",0,0,"get input signal","p f(d input)",PROCEDURE(p_get),},
{"getInputWeight",0,0,"get input weight","p f(d input)",PROCEDURE(p_getweight),},
{"getWeightedInputState",0,0,"get weighted input signal","p f(d input)",PROCEDURE(p_getw),},
{"getInputSum",0,0,"get signal sum","p f(d input)",PROCEDURE(p_getsum),},
{"getWeightedInputSum",0,0,"get weighted signal sum","p f(d input)",PROCEDURE(p_getwsum),"uses any number of inputs starting with the specified input. getWeightedInputSum(0)=weightedInputSum"},
{"getInputCount",0,0,"get input count","d",GETONLY(count),},
{"inputSum",0,0,"full signal sum","f",GETONLY(sum),},
{"weightedInputSum",0,0,"full weighted signal sum","f",GETONLY(wsum),},
{"getInputChannelCount",0,0,"get channel count for input","p d(d input)",PROCEDURE(p_getchancount),},
{"getInputStateChannel",0,0,"get input signal from channel","p f(d input,d channel)",PROCEDURE(p_getchan),},
{"getWeightedInputStateChannel",0,0,"get weighted input signal from channel","p f(d input,d channel)",PROCEDURE(p_getwchan),},
{"state",0,0,"neuron state (channel 0)","f",GETSET(state),"when read, returns the current neuron state.\nWhen written, sets the next neuron state (for use in the neuron definition)"},
{"channelCount",0,0,"number of output channels","d",GETSET(channels),},
{"getStateChannel",0,0,"get output state for channel","p f(d channel)",PROCEDURE(p_getstate),},
{"setStateChannel",0,0,"set output state for channel","p(d channel,f value)",PROCEDURE(p_setstate),},
{"hold",0,0,"Hold state","d 0 1",GETSET(hold),"",},
{"currState",0,0,"neuron state (channel 0)","f",GETSET(cstate),"the only difference from the \"state\" field is that currState, when written, changes the internal neuron state immediately (which disturbs the regular synchronous NN operation). This feature should only be used while controlling the neuron 'from outside' (like a neuro probe) and not in the neuron definition.",},
{"setCurrStateChannel",0,0,"set neuron for channel","p(d channel,f value)",PROCEDURE(p_setcstate),"like \"currState\""},
{"position_x",0,0,"position x","f",GETONLY(position_x),},
{"position_y",0,0,"position y","f",GETONLY(position_y),},
{"position_z",0,0,"position z","f",GETONLY(position_z),},
{"creature",0,0,"get owner creature","o Creature",GETONLY(creature),},
{"part",0,0,"the Part object where this neuron is located","o MechPart",GETONLY(part),},
{"joint",0,0,"the Joint object where this neuron is located","o MechJoint",GETONLY(joint),},
{"fields",0,0,"custom neuron fields","o Fields",GETONLY(fields),
"Neurons can have different fields depending on their class. Script neurons have their fields defined using the \"prop:\" syntax. If you develop a custom neuron script you should use the Fields object for accessing your own neuron fields. The Neuro.fields property is meant for accessing the neuron fields from the outside script.\n"
"Examples:\n"
"var c=LiveLibrary.createFromString(\"X[N]\");\n"
"Simulator.print(\"standard neuron inertia=\"+c.getNeuro(0).fields.in);\n"
"c=LiveLibrary.createFromString(\"X[Nn,e:0.1]\");\n"
"Simulator.print(\"noisy neuron error rate=\"+c.getNeuro(0).fields.e);\n"
"\n"
"The Interface object can be used to discover which fields are available for a certain neuron object:\n"
"c=LiveLibrary.createFromString(\"X[N]\");\n"
"var iobj=Interface.makeFrom(c.getNeuro(0).fields);\n"
"var i;\n"
"for(i=0;i<iobj.properties;i++)\n"
" Simulator.print(iobj.getId(i)+\" (\"+iobj.getName(i)+\")\");",},
{"def",0,0,"Neuron definition from which this live neuron was built","o NeuroDef",GETONLY(neurodef),},
#ifdef NEURO_SIGNALS
{"signals",0,PARAM_READONLY,"Signals","o NeuroSignals",FIELD(sigs_obj),},
#endif

{0,0,0,},
};
#undef FIELDSTRUCT

#ifdef NEURO_SIGNALS
ParamEntry neurosignals_paramtab[]=
 {
{"NeuroSignals",1,8,"NeuroSignals","Signals attached to the neuron.\nSee also: Signal, WorldSignals, CreatureSignals.\nscripts/light.neuro and scripts/seelight.neuro are simple custom neuron examples demonstrating how to send/receive signals between creatures.",},

#define FIELDSTRUCT NeuroSignals
{"add",0,PARAM_NOSTATIC,"add","p(s channel)",PROCEDURE(p_add),"Create a new signal"},
{"receive",0,PARAM_NOSTATIC,"receive","p f(s channel)",PROCEDURE(p_receive),"Receive the aggregated signal power in a given channel."},
{"receiveSet",0,PARAM_NOSTATIC,"receive","p oVector(s channel,f max distance)",PROCEDURE(p_receiveSet),"Get all signals in the specified range. Returns a readonly vector object containing Signal objects (individual signals can be accessed as result[0] throught result[result.size-1])."},
{"receiveFilter",0,PARAM_NOSTATIC,"receive","p f(s channel,f max distance,f flavor,f filter)",PROCEDURE(p_receiveFilter),"Receive the aggregated signal power in a given channel.\n\nAdditional filtering options:\n- Max distance only receives the neighbor signals (based on their physical location)\n- Flavor filtering: only signals having the flavor close to the specified one will be received. The filter value is the maximum allowed difference."},
{"receiveSingle",0,PARAM_NOSTATIC,"receive","p oSignal(s channel,f range)",PROCEDURE(p_receiveSingle),"Find the signal source having the highest signal power (including the distance)"},
#undef FIELDSTRUCT

#define FIELDSTRUCT SignalSet
{"get",0,PARAM_NOSTATIC,"get","p oSignal(d index)",PROCEDURE(p_get),},
{"size",0,1+PARAM_NOSTATIC,"size","d",GETONLY(size),},
{"clear",0,1+PARAM_NOSTATIC,"clear","p()",PROCEDURE(p_clear),},
#undef FIELDSTRUCT
{0,0,0,},
 }; 
Param neurosignals_param(neurosignals_paramtab,0);
#endif

Param st_neuroimplparam(neuroimpl_tab,0,"Neuro");

#ifdef NEURO_SIGNALS
class NeuroSigSource: public SigSource
{
  protected:
NeuroImpl* owner;
  public:
NeuroSigSource(NeuroImpl *n,Creature *c):SigSource(0,c),owner(n) {}
bool update();
};

bool NeuroSigSource::update()
{
Pt3D p;
if (owner->getPosition(p))
	{
	setLocation(p);
	return true;
	}
return false;
}

Creature *NeuroSignals::getCreature()
{
if (!cr)
	{
	cr=owner->getCreature();
	}
return cr;
}

void NeuroSignals::p_add(PARAMPROCARGS)
{
if (!owner->owner->channels) {ret->setEmpty();return;}
SigChannel *ch=owner->owner->channels->getChannel(args->getString(),true);
SigSource *s=new NeuroSigSource(owner,getCreature());
ch->addSource(s);
sigs+=s;
s->setupObject(ret);
}

void NeuroSignals::p_receive(PARAMPROCARGS)
{
SigChannel *ch; Pt3D p;
if (owner->owner->channels && (ch=owner->owner->channels->getChannel(args->getString(),false)) && owner->getPosition(p))
	ret->setDouble(ch->receive(&p,getCreature()));
else
	ret->setDouble(0);
}

void NeuroSignals::p_receiveFilter(PARAMPROCARGS)
{
SigChannel *ch; Pt3D p;
if (owner->owner->channels && (ch=owner->owner->channels->getChannel(args[3].getString(),false)) && owner->getPosition(p))
	ret->setDouble(ch->receive(&p,getCreature(),args[2].getDouble(),args[1].getDouble(),args[0].getDouble()));
else
	ret->setDouble(0);
}

void NeuroSignals::p_receiveSet(PARAMPROCARGS)
{
SigChannel *ch; Pt3D p;
SigVector *vec=new SigVector();
if (owner->owner->channels && (ch=owner->owner->channels->getChannel(args[1].getString(),false)) && owner->getPosition(p))
	ch->receiveSet(vec,&p,getCreature(),args[0].getDouble());
ret->setObject(vec->makeObject());
}

void NeuroSignals::p_receiveSingle(PARAMPROCARGS)
{
SigChannel *ch; Pt3D p;
if (owner->owner->channels && (ch=owner->owner->channels->getChannel(args[1].getString(),false)) && owner->getPosition(p))
	{
	SigSource *src=ch->receiveSingle(&p,getCreature(),args[0].getDouble(),0,1e99);
	if (src)
		{
		src->setupObject(ret);
		return;
		}
	}
ret->setEmpty();
}
#endif

extern ParamEntry creature_paramtab[];
static Param creature_param(creature_paramtab,0);

Creature* NeuroImpl::getCreature()
{
#ifndef NOCREATUREOBJECT
CreatMechObject *cmo=(CreatMechObject *)neuro->owner->userdata[CreatMechObject::modeltags_id];
return cmo->creature;
#else
return 0;
#endif
}

void NeuroImpl::get_creature(ExtValue *ret)
{
#ifndef NOCREATUREOBJECT
ret->setObject(ExtObject(&creature_param,getCreature()));
#endif
}

void NeuroImpl::get_part(ExtValue *ret)
{
#ifndef NOCREATUREOBJECT
Part *pa;
if (pa=neuro->getPart())
	ret->setObject(ExtObject(&mechpart_param,((MechPart *)pa->userdata[CreatMechObject::modeltags_id])));
else
	ret->setEmpty();
#endif
}

void NeuroImpl::get_joint(ExtValue *ret)
{
#ifndef NOCREATUREOBJECT
Joint *jo;
if (jo=neuro->getJoint())
	ret->setObject(ExtObject(&mechjoint_param,((MechJoint*)jo->userdata[CreatMechObject::modeltags_id])));
else
	ret->setEmpty();
#endif
}

bool NeuroImpl::getPosition(Pt3D &pos)
{
#ifndef NOCREATUREOBJECT
Part *pa; Joint *jo;
if (pa=neuro->getPart())
	{pos=((MechPart *)pa->userdata[CreatMechObject::modeltags_id])->p; return true;}
if (jo=neuro->getJoint())
	{
	if (neuro->getClass()->getVisualHints() & NeuroClass::AtFirstPart)
		pos=((MechPart*)jo->part1->userdata[CreatMechObject::modeltags_id])->p;
	else if (neuro->getClass()->getVisualHints() & NeuroClass::AtSecondPart)
		pos=((MechPart*)jo->part2->userdata[CreatMechObject::modeltags_id])->p;
	else pos=(((MechPart*)jo->part1->userdata[CreatMechObject::modeltags_id])->p
		   +((MechPart*)jo->part2->userdata[CreatMechObject::modeltags_id])->p)/2;
	return true;
	}
#endif
return false;
}

void NeuroImpl::get_position_x(ExtValue *ret)
{Pt3D pos; if (getPosition(pos)) ret->setDouble(pos.x); else ret->setEmpty();}
void NeuroImpl::get_position_y(ExtValue *ret)
{Pt3D pos; if (getPosition(pos)) ret->setDouble(pos.y); else ret->setEmpty();}
void NeuroImpl::get_position_z(ExtValue *ret)
{Pt3D pos; if (getPosition(pos)) ret->setDouble(pos.z); else ret->setEmpty();}


void NeuroImpl::createFieldsObject()
{
fields_param=new Param(paramentries?paramentries:(ParamEntry*)&empty_paramtab,this,"Fields");
fields_object=new ExtObject(fields_param);
}

void NeuroImpl::get_fields(ExtValue *ret)
{
if (!fields_object)
	createFieldsObject();
ret->setObject(*fields_object);
}

void NeuroImpl::get_neurodef(ExtValue *ret)
{
ret->setObject(ExtObject(&st_neuroparam,neuro));
}

NeuroImpl::~NeuroImpl()
{
if (fields_param)
	{
	delete fields_param;
	delete fields_object;
	}
}
