// This file is a part of the Framsticks GDK.
// Copyright (C) 1999-2014  Maciej Komosinski and Szymon Ulatowski.  See LICENSE.txt for details.
// Refer to http://www.framsticks.com/ for further information.

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <frams/virtfile/stdiofile.h>

#include <frams/model/model.h>
#include <frams/errmgr/stdouterr.h>
#include <frams/genetics/preconfigured.h>

/**
 @file
 Sample code: Convert old-style shapes (sticks are Joints) to new style (sticks are Parts)
 Usage: shapeconvert [-sSHAPE] [-tTHICKNESS] [genotype_or_stdin]

 Calling examples:

 # convert any genotype to f0 using new-style shapes:
 shapeconvert "X"

 //0
 m:
 p:0.5, sh=3, sx=0.5, sy=0.200000002980232, sz=0.200000002980232, rx=-1.5707963267949, rz=3.14159265358979

 # load a genotype from file, convert to new shapes, display:
 loader_test data/other.gen "Bird" | shapeconvert -s1 - | theater.exe -g -

 */

int main(int argc, char*argv[])
{
	StdioFILE::setStdio();//setup VirtFILE::Vstdin/out/err
	StdoutErrorHandler err(ErrorHandlerBase::DontBlock, VirtFILE::Vstderr); //errors -> stderr, don't interfere with stdout

	PreconfiguredGenetics genetics;
	Part::Shape shape = Part::SHAPE_CYLINDER;
	float thickness = 0.2;

	char* gen_arg = 0;
	for (int i = 1; i < argc; i++)
	{
		char* ar = argv[i];
		if (ar[0] == '-')
			switch (ar[1])
		{
			case 's': shape = (Part::Shape)atol(ar + 2);
				if ((shape != Part::SHAPE_ELLIPSOID) && (shape != Part::SHAPE_CUBOID) && (shape != Part::SHAPE_CYLINDER))
				{
					FMprintf("", "shapeconvert", FMLV_ERROR, "Invalid shape");
					return 4;
				}
				break;
			case 't': thickness = atof(ar + 2); break;
			case 'h': puts("Usage: shapeconvert [-sSHAPE] [-tTHICKNESS] [genotype_or_stdin]\n\tSHAPE: 1=ellipsoid, 2=cuboid, 3(default)=cylinder\n\tTHICKNESS: used for Part.sy/sz (default=0.2)"); break;
		}
		else
			if (!gen_arg)
				gen_arg = ar;
	}
	SString gen;
	if (gen_arg)
		gen = gen_arg;
	else
		loadSString(VirtFILE::Vstdin, gen);
	Geno g(gen);
	Model m(g);

	if (!m.isValid())
	{
		FMprintf("", "shapeconvert", FMLV_ERROR, "Cannot build Model from the supplied genotype");
		return 2;
	}

	if (m.getShape() != Model::SHAPE_OLD)
	{
		FMprintf("", "shapeconvert", FMLV_ERROR, "Only old style shapes can be converted");
		return 3;
	}

	Model newmodel;
	newmodel.open();
	newmodel.buildUsingNewShapes(m, shape, thickness);
	newmodel.close();

	// Normal f0 would be retrieved using m.getF0Geno().
	// The following form allows for more control:
	Geno f0_g = newmodel.getF0Geno();
	puts((const char*)f0_g.toString());

	return 0;
}
