// This file is a part of the Framsticks GDK.
// Copyright (C) 1999-2014  Maciej Komosinski and Szymon Ulatowski.  See LICENSE.txt for details.
// Refer to http://www.framsticks.com/ for further information.

#include "stl-util.h"
#include <stdarg.h>
#include <stdlib.h>
#include "nonstd_stdio.h"
#include "nonstd.h"
#include "framsg.h"
#include <assert.h>

string ssprintf_va(const char* format, va_list ap)
{
	string s; //clang crashed when this declaration was in s=buf
	long size = 256;
	char* buf;

	//almost like SString::sprintf, but there is no common code to share because SString can use its directWrite to avoid double allocating/copying
#ifdef USE_VSCPRINTF
	size = _vscprintf(format, ap) + 1; //+1 for terminating null character
#endif

	while (1)
	{
		buf = (char*)malloc(size);
		assert(buf != NULL);
		int n = vsnprintf(buf, size, format, ap);
		if (n > -1 && n < size)
		{
			s = buf;
			free(buf);
			return s;
		}
#ifdef VSNPRINTF_RETURNS_REQUIRED_SIZE
		if (n > -1)    /* glibc 2.1 */
			size = n+1; /* precisely what is needed */
		else           /* glibc 2.0 */
#endif
			size *= 2;  /* twice the old size */
		free(buf);
	}
}

string ssprintf(const char* format, ...)
{
	va_list ap;
	va_start(ap, format);
	string ret = ssprintf_va(format, ap); //is it too wasteful? copying the string again... unless the compiler can handle it better
	va_end(ap);
	return ret;
}

bool readCompleteFile(const char* filename, vector<char>& data, bool warn_on_missing_file)
{
	MFILE *f = mfopen(filename, FOPEN_READ_BINARY);
	bool ok = f != NULL;
	if (f)
	{
		mfseek(f, 0, SEEK_END);
		long size = mftell(f);
		mfseek(f, 0, SEEK_SET);
		data.resize(size);
		int przeczytane = mfread(&data[0], size, 1, f);
		mfclose(f);
		ok &= przeczytane == 1;
	}
	if (warn_on_missing_file && !ok)
		FMprintf("stl-util", "readCompleteFile", FMLV_WARN, "Couldn't open file '%s'", filename);
	return ok;
}

bool readCompleteFile(const char* filename, string& out, bool warn_on_missing_file)
{
	vector<char> data;
	if (readCompleteFile(filename, data, warn_on_missing_file))
	{
		out = string(&data[0], data.size());
		return true;
	}
	return false;
}

bool writeCompleteFile(const char* filename, const string& text, bool warn_on_fail)
{
	MFILE *f = mfopen(filename, FOPEN_WRITE_BINARY);
	bool ok = f != NULL;
	if (f)
	{
		int zapisane = mfwrite(text.c_str(), text.length(), 1, f);
		mfclose(f);
		ok &= zapisane == 1;
	}
	if (warn_on_fail && !ok)
		FMprintf("stl-util", "writeCompleteFile", FMLV_WARN, "couldn't write file '%s'", filename);
	return ok;
}

bool writeCompleteFile(const char* filename, vector<char>& data, bool warn_on_fail)
{
	string s(&data[0], data.size());
	return writeCompleteFile(filename, s, warn_on_fail);
}



string stripExt(const string& filename)
{
	int dot = filename.rfind('.');
	if (dot == string::npos) return filename;
	int sep = filename.rfind(PATH_SEPARATOR_CHAR);
	if ((sep == string::npos) || (sep < dot))
		return filename.substr(0, dot);
	return filename;
}

string getFileExt(const string& filename)
{
	int dot = filename.rfind('.');
	if (dot == string::npos) return string("");
	int sep = filename.rfind(PATH_SEPARATOR_CHAR);
	if ((sep == string::npos) || (sep < dot))
		return filename.substr(dot);
	return string("");
}

string getFileDir(const string& filename)
{
	int slash = filename.rfind(PATH_SEPARATOR_CHAR);
	if (slash == string::npos) return string("");
	return filename.substr(0, slash);
}
