// This file is a part of the Framsticks GDK.
// Copyright (C) 2002-2014  Maciej Komosinski and Szymon Ulatowski.  See LICENSE.txt for details.
// Refer to http://www.framsticks.com/ for further information.

#include "collectionobj.h"
#include <common/nonstd_math.h> //sqrt in borland
#include <frams/errmgr/stderrors.h>
#include <common/nonstd_stl.h>
#include <frams/util/sstringutils.h>
#ifndef NO_VMACHINE
#include <frams/vm/vmachine.h>
#endif

#define FIELDSTRUCT VectorObject
ParamEntry vector_paramtab[]=
{
{"Vector",1,13,"Vector","Vector is 1-dimensional array, indexed by integer value (starting from 0). "
 "Multidimensional arrays can be simulated by putting other Vector objects into the Vector.\n"
 "Example:\nvar v=Vector.new();\nv.add(123); v.add(\"string\");",},
{"clear",0,PARAM_NOSTATIC,"clear data","p()",PROCEDURE(p_clear),},
{"size",0,PARAM_READONLY+PARAM_NOSTATIC,"element count","d",GETONLY(size),},
{"remove",0,PARAM_NOSTATIC,"remove at position","p(d position)",PROCEDURE(p_remove),},
{"get",0,PARAM_NOSTATIC,"get value at position","p x(d position)",PROCEDURE(p_get),},
{"set",0,PARAM_NOSTATIC,"set value at position","p(d position,x value)",PROCEDURE(p_set),},
{"add",0,PARAM_NOSTATIC,"append at the end","p(x value)",PROCEDURE(p_add),},
{"find",0,PARAM_NOSTATIC,"find","p d(x value)",PROCEDURE(p_find),"returns the element index or -1 if not found"},
{"avg",0,PARAM_READONLY+PARAM_NOSTATIC,"average","f",GETONLY(avg)},
{"stdev",0,PARAM_READONLY+PARAM_NOSTATIC,"standard deviation","f",GETONLY(stdev),"=sqrt(sum((element[i]-avg)^2)/(size-1)) which is estimated population std.dev. from sample std.dev."},
{"toString",0,PARAM_READONLY+PARAM_NOSTATIC,"textual form","s",GETONLY(toString),},
{"new",0,0,"create new Vector","p oVector()",STATICPROCEDURE(p_new),},
{"sort",0,PARAM_NOSTATIC,"sort elements (in place)","p(o comparator)",PROCEDURE(p_sort),"comparator can be null, giving the \"natural\" sorting order (depending on element type), otherwise it must be a function reference obtained by the \"function FUNCTIONNAME\" operator.\n\nExample:\nfunction compareLastDigit(a,b) {return (a%10)<(b%10);}\nvar v=[16,23,35,42,54,61];\nv.sort(function compareLastDigit);"},
{"iterator",0,PARAM_NOSTATIC+PARAM_READONLY,"iterator","o",GETONLY(iterator),},
{0,0,0,},
};
#undef FIELDSTRUCT

#define FIELDSTRUCT DictionaryObject
ParamEntry dictionary_paramtab[]=
{
{"Dictionary",1,9,"Dictionary","Dictionary associates stored values with string keys "
 "(\"key\" is the first argument in get/set/remove functions). Integer \"key\" can be "
 "used to enumerate all elements (the sequence of elements is not preserved).\n"
 "Example:\nvar d=Dictionary.new(); d.set(\"name\",\"John\"); d.set(\"age\",44);\n"
 "var i;\nfor(i=0;i<d.size;i++) Simulator.print(d.getKey(i)+\" is \"+d.get(i));",},
{"clear",0,PARAM_NOSTATIC,"clear data","p()",PROCEDURE(p_clear),},
{"size",0,PARAM_NOSTATIC+PARAM_READONLY,"element count","d",GETONLY(size),},
{"remove",0,PARAM_NOSTATIC,"remove named or indexed element","p(x key)",PROCEDURE(p_remove),},
{"get",0,PARAM_NOSTATIC,"get named or indexed element","p x(x key)",PROCEDURE(p_get),},
{"getKey",0,PARAM_NOSTATIC,"get a key of the indexed element","p s(d index)",PROCEDURE(p_getKey),},
{"set",0,PARAM_NOSTATIC,"set named or indexed element","p(x key,x value)",PROCEDURE(p_set),},
{"find",0,PARAM_NOSTATIC,"find","p s(x value)",PROCEDURE(p_find),"returns the element key or null if not found"},
{"new",0,0,"create new Dictionary","p oDictionary()",STATICPROCEDURE(p_new),},
{"toString",0,PARAM_READONLY+PARAM_NOSTATIC,"textual form","s",GETONLY(toString),},
{0,0,0,},
};
#undef FIELDSTRUCT

Param VectorObject::par(vector_paramtab);
Param DictionaryObject::par(dictionary_paramtab);

/////////////////////////////////////////

VectorObject::VectorObject(Pt3D &pt)
	:readonly(0),owndata(1)
{
set(0,ExtValue(pt.x));
set(1,ExtValue(pt.y));
set(2,ExtValue(pt.z));
}

void VectorObject::clear()
{
if (owndata)
for(int i=data.size()-1;i>=0;i--)
	{
	ExtValue *v=(ExtValue*)data.get(i);
	if (v) delete v;
	}
data.clear();
}

void VectorObject::p_remove(PARAMPROCARGS)
{
if (readonly) return;
int i=args->getInt();
if ((i<0)||(i>=data.size())) return;
ExtValue *v=(ExtValue*)data.get(i);
if (v) delete v;
data-=i;
}

void VectorObject::set(int i,const ExtValue& val)
{
int oldsize=data.size();
if (i<0) return;
ExtValue *v=(ExtValue*)data.get(i);
if (v) delete v;
data.set(i,new ExtValue(val));
i--;
while(i>=oldsize)
	{
	data.set(i,0);
	i--;
	}
}

void VectorObject::p_get(PARAMPROCARGS)
{
int i=args->getInt();
if (listIndexCheck(&data,i,"VectorObject","get"))
	{
	ExtValue *v=get(i);
	if (v)
		{
		*ret=*v;
		return;
		}
	}
*ret=ExtValue();
}

void VectorObject::get_avg(ExtValue* ret)
{
if (!data.size()) {ret->setEmpty(); return;}
double s=0.0;
for(int i=data.size()-1;i>=0;i--)
	s+=((ExtValue*)data.get(i))->getDouble();
s/=data.size();
ret->setDouble(s);
}

SString VectorObject::serialize() const
{
SString out="[";
	{
	for(int i=0;i<data.size();i++)
		{
		ExtValue* v=(ExtValue*)data.get(i);
		if (i) out+=",";
		if (v)
			out+=v->serialize();
		else
			out+="null";
		}
	}
out+="]";
//sprintf(out.directAppend(20),"<Vector@%p>",this);out.endAppend();
return out;
}

static THREAD_LOCAL_DEF(SList,tostring_trace);

void VectorObject::get_toString(ExtValue* ret)
{
SString out="[";
//static SListTempl<VectorObject*> trace;
if (tlsGetRef(tostring_trace).find(this)>=0)
	out+="...";
else
	{
	tlsGetRef(tostring_trace)+=this;
	for(int i=0;i<data.size();i++)
		{
		ExtValue* v=(ExtValue*)data.get(i);
		if (i) out+=",";
		if (v)
			out+=v->getString();
		else
			out+="null";
		}
	tlsGetRef(tostring_trace)-=this;
	}
out+="]";
ret->setString(out);
}

void VectorObject::get_stdev(ExtValue* ret)
{
if (!data.size()) {ret->setEmpty(); return;}
get_avg(ret);
double a=ret->getDouble();
double s=0.0;
for(int i=data.size()-1;i>=0;i--)
	{
	double d=a-((ExtValue*)data.get(i))->getDouble();
	s+=d*d;
	}
ret->setDouble(sqrt(s/max(1,data.size()-1)));
}

void VectorObject::p_find(PARAMPROCARGS)
{
short i;
for(i=0;i<data.size();i++)
	{
	if ((*args)==(*get(i)))
		{ret->setInt(i);return;}
	}
ret->setInt(-1);
}

class VEComparator
{
public:
bool operator()(const ExtValue *a,const ExtValue *b) {return a->compare(*b)<0;}
};

#ifndef NO_VMACHINE
class VMVEComparator
{
public:
VMachine::JumpTargetObject *jto;
VMachine *vm;
VMVEComparator(VMachine::JumpTargetObject *_jto):jto(_jto),vm(jto->vm) {}
bool operator()(const ExtValue *a,const ExtValue *b);
};

bool VMVEComparator::operator()(const ExtValue *a,const ExtValue *b)
{
if (!VMCode::prepareDynamicJumpTarget(jto->pc,jto->code))
	return false;

vm->push(*a);
vm->push(*b);
vm->pushNewCallState();
vm->jumpDynamicJumpTarget(jto->pc);
vm->run();
vm->popCallState();
bool ret;
ExtValue& retval=vm->getValue();
if (retval.type==TInvalid)
	{
	ret=false;
	FMprintf("VectorElementComparator","",FMLV_ERROR,"Comparison function returned no value");
	}
else
	ret=(retval.getInt()!=0);
vm->drop(2);
return ret;
}
#endif

void VectorObject::p_sort(PARAMPROCARGS)
{
#ifndef NO_VMACHINE
VMachine::JumpTargetObject *jto=VMachine::JumpTargetObject::fromObject(args->getObject());
if (jto)
	{
	VMVEComparator cmp(jto);
	ExtValue **first=(ExtValue**)&data.getref(0);
	std::sort(first,first+data.size(),cmp);
	}
else
#endif
	{
	VEComparator cmp;
	ExtValue **first=(ExtValue**)&data.getref(0);
	std::sort(first,first+data.size(),cmp);
	}
ret->setEmpty();
}

void VectorObject::get_iterator(ExtValue* ret)
{
ret->setObject(VectorIterator::makeFrom(this));
}

VectorObject* VectorObject::fromObject(const ExtObject& o)
{
return (VectorObject*)o.getTarget(par.getName());
}

/////////////////////////////

void DictionaryObject::clear()
{
for(HashEntryIterator it(hash);it.isValid();)
	{
        ExtValue *v=(ExtValue*)hash.remove(it);
	if (v) delete v;
	}
hash.clear();
hash.init();
}

void DictionaryObject::p_find(PARAMPROCARGS)
{
for(HashEntryIterator it(hash);it.isValid();it++)
	{
	if ((*args)==(*((ExtValue*)it->value)))
		{
		ret->setString(it->key);
		return;
		}
	}
ret->setEmpty();
}

HashEntryIterator* DictionaryObject::getIndexIterator(int i)
{
if (i<0) return 0;
if (i>=hash.getSize()) return 0;

if ((!it.isValid())||(it_index>i))
	{
	it=HashEntryIterator(hash);
	it_index=0;
	}
while(it.isValid())
	{
	if (it_index==i)
		return &it;
	it_index++;
	it++;
	}
return 0;
}

void DictionaryObject::p_remove(PARAMPROCARGS)
{
if ((args->type==TInt)||(args->type==TDouble))
	{
	HashEntryIterator* iter=getIndexIterator(args->getInt());
	if (iter)
		{
		ExtValue *oldval=(ExtValue*)hash.remove(*iter);
		if (oldval) {*ret=*oldval; delete oldval;} else *ret=ExtValue();
		}
	}
else
	{
	ExtValue *oldval=(ExtValue*)hash.remove(args[0].getString());
	if (oldval) {*ret=*oldval; delete oldval;} else *ret=ExtValue();
	}
}

void DictionaryObject::p_get(PARAMPROCARGS)
{
if ((args->type==TInt)||(args->type==TDouble))
	{
	HashEntryIterator* iter=getIndexIterator(args->getInt());
	if (iter && (*iter)->value)
		{
		*ret=*((ExtValue*)(*iter)->value);
		return;
		}
	}
else
	{
	ExtValue *val=(ExtValue*)hash.get(args[0].getString());
	if (val)
		{
		*ret=*val;
		return;
		}
	}
*ret=ExtValue();
}

void DictionaryObject::p_getKey(PARAMPROCARGS)
{
HashEntryIterator* iter=getIndexIterator(args->getInt());
if (iter)
	{
	*ret=(*iter)->key;
	return;
	}
*ret=ExtValue();
}

void DictionaryObject::p_set(PARAMPROCARGS)
{
ExtValue *newval=(args[0].getType()==TUnknown)?0:new ExtValue(args[0]);
ExtValue *oldval=(ExtValue*)hash.put(args[1].getString(),newval);
if (oldval) {*ret=*oldval; delete oldval;} else *ret=ExtValue();
}

SString DictionaryObject::serialize() const
{
SString out="{";
	{
	for(HashEntryIterator it(hash);it.isValid();)
		{
		out+="\"";
		SString q=it->key; sstringQuote(q);
		out+=q;
		out+="\":";
		if (it->value!=NULL)
			out+=((ExtValue*)it->value)->serialize();
		else
			out+="null";
		it++;
		if (it.isValid()) out+=",";
		}
	}
out+="}";
return out;
}

void DictionaryObject::get_toString(ExtValue* ret)
{
SString out="{";
//static SListTempl<DictionaryObject*> trace;
if (tlsGetRef(tostring_trace).find(this)>=0)
	out+="...";
else
	{
	tlsGetRef(tostring_trace)+=this;
	for(HashEntryIterator it(hash);it.isValid();)
		{
		out+=it->key;
		out+=":";
		if (it->value!=NULL)
			out+=((ExtValue*)it->value)->getString();
		else
			out+="null";
		it++;
		if (it.isValid()) out+=",";
		}
	tlsGetRef(tostring_trace)-=this;
	}
out+="}";
ret->setString(out);
}

DictionaryObject* DictionaryObject::fromObject(const ExtObject& o)
{
return (DictionaryObject*)o.getTarget(par.getName());
}

////////////////

VectorIterator::VectorIterator(VectorObject* v)
{
vec=v;
vec->incref();
pos=-1;
}

#define FIELDSTRUCT VectorIterator
ParamEntry vectoriterator_paramtab[]=
{
 {"VectorIterator",1,2,"VectorIterator","VectorIterator"},
{"next",0,PARAM_READONLY+PARAM_NOSTATIC,"next","d 0 1",GETONLY(next),},
{"value",0,PARAM_READONLY+PARAM_NOSTATIC,"value","x",GETONLY(value),},
{0,0,0,},
};
#undef FIELDSTRUCT

ExtObject VectorIterator::makeFrom(VectorObject *v)
{
static Param par(vectoriterator_paramtab);
return ExtObject(&par,new VectorIterator(v));
}

VectorIterator::~VectorIterator()
{
vec->decref();
}

void VectorIterator::get_next(ExtValue* ret)
{
pos++;
ret->setInt((pos < vec->data.size()) ? 1 : 0);
}

void VectorIterator::get_value(ExtValue* ret)
{
ExtValue *v=(ExtValue*) (((pos>=0)&&(pos<vec->data.size())) ? vec->data(pos) : NULL );
if (v)
	*ret=*v;
else
	ret->setEmpty();
}
