/**
 * @file Simulation Module
 * @author Patryk Gliszczynski, with further modifications
 * @version 1.0
 */

class Simulation {
  /**
   * Class reponsible for the control of simulation flow.
   */

  constructor(world, args) {
    this.world = world;
	this.buttons = [];

    this.genPoolText = this.getText("Genotype Pool", EVOLUTION_METHOD_TEXT_POSITION);
    this.mutationText = this.getText("Mutation", EVOLUTION_METHOD_TEXT_POSITION);
    this.crossoverText = this.getText("Crossover", EVOLUTION_METHOD_TEXT_POSITION);
    this.cloningText = this.getText("Cloning", EVOLUTION_METHOD_TEXT_POSITION);
    this.mutatedText = this.getText("Mutated", EVOLUTION_METHOD_TEXT_POSITION);
    this.offspringText = this.getText("Offspring", EVOLUTION_METHOD_TEXT_POSITION);
    this.clonedText = this.getText("Cloned", EVOLUTION_METHOD_TEXT_POSITION);
    this.fitnessText = this.getText("Fitness", Config.Simulation.Element.Position.FITNESS_TEXT);

	this.genotypeEncoding = Config.Simulation.GENOTYPE;
	this.proposedGenotypeEncoding = undefined;

    this.world.scene.add(this.genPoolText);
    this.world.scene.add(this.mutationText);
    this.world.scene.add(this.crossoverText);
    this.world.scene.add(this.cloningText);
    this.world.scene.add(this.mutatedText);
    this.world.scene.add(this.offspringText);
    this.world.scene.add(this.clonedText);
    this.world.scene.add(this.fitnessText);

	this.world.createTable(this.createGenotypes(Config.Simulation.NUMBER_OF_GENOTYPES,Config.Simulation.INITIAL_MUTATIONS),this.genotypeEncoding);

    this.state = State.INITIAL;
    this.timing = Config.Simulation.Timing.INITIAL;

	this.mutationProb = 0;
	this.mutationListeners = [];
	this.setMutationProb(args.mutation_prob?args.mutation_prob:Config.Simulation.MUTATION_PROBABILITY);
	
	this.crossoverProb = 0;
	this.crossoverListeners = [];
	this.setCrossoverProb(args.crossover_prob?args.crossover_prob:Config.Simulation.CROSSOVER_PROBABILITY);

	this.neuroLayoutModel = undefined;
	this.neuroLayoutFunctionHelper = new Module.NNLayoutFunctionHelper();
	this.neuroViewer = undefined;

	this.evolvedFitness = 0;

	this.lastTimeout = undefined;
  }

  activateButtons(){
	for(let i=0;i<this.buttons.length;i++){
		this.buttons[i].activate();
	}
  }

  deactivateButtons(){
	for(let i=0;i<this.buttons.length;i++){
		this.buttons[i].deactivate();
	}
  }

  updateListeners(list,value){
	for(let i=0;i<list.length;i++){
		list[i].update(value);
	}
  }

  reupdateListeners(){
	this.updateListeners(this.mutationListeners,this.mutationProb);
	this.updateListeners(this.crossoverListeners,this.crossoverProb);
  }

  getMutationProb(){
	return this.mutationProb;
  }
  setMutationProb(prob){
	this.mutationProb = Math.min(Math.max(prob,0),1);
	if(this.crossoverProb + this.mutationProb > 1){
		this.setCrossoverProb(1 - this.mutationProb);
	}
	this.updateListeners(this.mutationListeners,this.mutationProb);
  }

  getCrossoverProb(){
	return this.crossoverProb;
  }
  setCrossoverProb(prob){
	this.crossoverProb = Math.min(Math.max(prob,0),1);
	if(this.crossoverProb + this.mutationProb > 1){
		this.setMutationProb(1 - this.crossoverProb);
	}
	this.updateListeners(this.crossoverListeners,this.crossoverProb);
  }

  updateNeuroLayoutModel(model) {
    if( this.neuroLayoutModel ) {
      Module.destroy( this.neuroLayoutModel );
    }
    this.neuroLayoutModel = new Module.NNLayoutState_Model_Fred( model );
    this.neuroLayoutFunctionHelper.doLayout( Config.NeuroViewer.layoutType , this.neuroLayoutModel );
  }

  getBrainData (model) {
	var thisSim = this;
    return {
      getElements: function () {
        return thisSim.neuroLayoutModel.GetElements();
      },
      getValueXYWH: function ( i ) {
        return thisSim.neuroLayoutModel.GetValueXYWH( i );
      },
      getNeuro: function ( i ) {
        return model.getNeuro( i );
      }
    };
  }

  updateFitness(value){
	// if (this.state !== State.SHOW_FRAMSTICK && this.state !== State.DESTROY_FRAMSTICK){
	// 	return;
	// }
	this.evolvedFitness += value;
	this.evolvedFitness = Math.max(this.evolvedFitness,0);
	if(this.evolvedFitnessPlank){
		this.evolvedFitnessPlank.updateFitness(this.evolvedFitness);
	}
  }

  isGenotypeChanging(){
	return this.proposedGenotypeEncoding && this.proposedGenotypeEncoding !== this.genotypeEncoding;
  }

  run() {
    switch (this.state) {

      case State.INITIAL:
        this.showText(this.genPoolText);

        this.state = State.EVOLUTION_METHOD_SELECTION;
        this.timing = Config.Simulation.Timing.INITIAL;

		if(this.isGenotypeChanging()){
			this.state = State.CHANGE_GENOTYPE;
		}

        break;

      case State.EVOLUTION_METHOD_SELECTION:
        this.moveCamera(Config.Simulation.Camera.Position.TABLE);

        this.timing = Config.Simulation.Timing.EVOLUTION_METHOD_SELECTION;
        let randomVal = Math.random()
        if (randomVal < this.mutationProb) {
          this.state = State.MUTATION_GENOTYPE_SELECTION;
        } else if (randomVal < this.mutationProb +
                               this.crossoverProb){
          this.state = State.CROSSOVER_GENOTYPE_SELECTION;
        } else {
          this.state = State.CLONING_GENOTYPE_SELECTION;
        }
        this.evolutionMethod = this.state;
        break;

      case State.MUTATION_GENOTYPE_SELECTION:
        this.showText(this.mutationText)
        this.hideText(this.genPoolText);

        this.mutationIdx = this.chooseIdx();
        this.mutationArrow = this.getArrow(this.mutationIdx).mesh;
        this.world.scene.add(this.mutationArrow);

        this.timing = Config.Simulation.Timing.GENOTYPE_SELECTION;
        this.state = State.MUTATION_RESULT;
        break;

      case State.MUTATION_RESULT:
        this.evolvedGenotype = this.getMutatedGenotype(this.world.table.genotypes[this.mutationIdx]);
        this.evolvedFitness = Math.max(0, (parseFloat(this.world.table.fitnessPlanks[this.mutationIdx].fitness) +
                                           (Math.random() * 20 - 10)).toFixed(2));
        this.showEvolutionResult();

        this.timing = Config.Simulation.Timing.EVOLUTION_RESULT;
        this.state = State.SHOW_FRAMSTICK;
        break;

      case State.MUTATION_RETURN:
        this.moveCamera(Config.Simulation.Camera.Position.TABLE);
        this.evolutionReturn()

        this.hideText(this.mutationText);
        this.hideText(this.mutatedText);
        this.world.scene.remove(this.mutationArrow);
        this.world.scene.remove(this.evolvedFramstick);

        this.timing = Config.Simulation.Timing.EVOLUTION_RETURN;
        this.state = State.INITIAL;
        break;

      case State.CLONING_GENOTYPE_SELECTION:
        this.showText(this.cloningText);
        this.hideText(this.genPoolText);

        this.cloningIdx = this.chooseIdx();
        this.cloningArrow = this.getArrow(this.cloningIdx).mesh;
        this.world.scene.add(this.cloningArrow);

        this.timing = Config.Simulation.Timing.GENOTYPE_SELECTION;
        this.state = State.CLONING_RESULT;
        break;

      case State.CLONING_RESULT:
        this.evolvedGenotype = this.world.table.genotypes[this.cloningIdx];
        this.evolvedFitness = Math.max(0, (parseFloat(this.world.table.fitnessPlanks[this.cloningIdx].fitness) +
                                           (Math.random() * 2 - 1)).toFixed(2));
        this.showEvolutionResult();

        this.timing = Config.Simulation.Timing.EVOLUTION_RESULT;
        this.state = State.SHOW_FRAMSTICK;
        break;

      case State.CLONING_RETURN:
        this.moveCamera(Config.Simulation.Camera.Position.TABLE);
        this.evolutionReturn()

        this.hideText(this.cloningText);
        this.hideText(this.clonedText);
        this.world.scene.remove(this.cloningArrow);
        this.world.scene.remove(this.evolvedFramstick);

        this.timing = Config.Simulation.Timing.EVOLUTION_RETURN;
        this.state = State.INITIAL;
        break;

      case State.CROSSOVER_GENOTYPE_SELECTION:
        this.showText(this.crossoverText)
        this.hideText(this.genPoolText);

        this.crossoverIdx1 = this.chooseIdx();
        do {
          this.crossoverIdx2 = this.chooseIdx();
        } while (this.crossoverIdx1 == this.crossoverIdx2);

        this.crossoverArrow1 = this.getArrow(this.crossoverIdx1).mesh;
        this.world.scene.add(this.crossoverArrow1);
        this.crossoverArrow2 = this.getArrow(this.crossoverIdx2).mesh;
        this.world.scene.add(this.crossoverArrow2);

        this.timing = Config.Simulation.Timing.GENOTYPE_SELECTION;
        this.state = State.CROSSOVER_RESULT;
        break;

      case State.CROSSOVER_RESULT:
        this.evolvedGenotype = this.getCrossoveredGenotype(this.world.table.genotypes[this.crossoverIdx1],
                                                           this.world.table.genotypes[this.crossoverIdx2]);
        this.evolvedFitness = ((parseFloat(this.world.table.fitnessPlanks[this.crossoverIdx1].fitness) +
                               parseFloat(this.world.table.fitnessPlanks[this.crossoverIdx2].fitness)) / 2).toFixed(2);
        this.showEvolutionResult();

        this.timing = Config.Simulation.Timing.EVOLUTION_RESULT;
        this.state = State.SHOW_FRAMSTICK;
        break;

      case State.CROSSOVER_RETURN:
        this.moveCamera(Config.Simulation.Camera.Position.TABLE);
        this.evolutionReturn()

        this.hideText(this.crossoverText);
        this.hideText(this.offspringText);
        this.world.scene.remove(this.crossoverArrow1);
        this.world.scene.remove(this.crossoverArrow2);
        this.world.scene.remove(this.evolvedFramstick.mesh);

        this.timing = Config.Simulation.Timing.EVOLUTION_RETURN;
        this.state = State.INITIAL;
        break;

      case State.SHOW_FRAMSTICK:
		this.activateButtons();
        this.moveCamera(Config.Simulation.Camera.Position.FRAMSTICK);
		
        this.evolvedFramstick = new Framstick(this.evolvedGenotype);
		const model = this.evolvedFramstick.getModelFromGenotype(this.evolvedGenotype);
        this.world.scene.add(this.evolvedFramstick.mesh);
		
		this.updateNeuroLayoutModel(model);
		this.neuroViewer.updateBrain(this.getBrainData(model));
		Module.destroy(model);
        this.timing = Config.Simulation.Timing.SHOW_FRAMSTICK;
        this.state = State.DESTROY_FRAMSTICK;
        break;

      case State.DESTROY_FRAMSTICK:
        this.evolvedFramstick.scaleUpSize = 0.01;
        this.evolvedFramstick.scaleDownSize = 0.01;
		this.neuroViewer.clear();
		this.world.scene.remove(this.neuroViewer.get3DContainer());

        this.showText(this.fitnessText);
        this.rotateText(this.fitnessText, Config.Simulation.Element.Angle.SIDE_VIEW);
        this.evolvedFitnessPlank = new FitnessPlank(this.evolvedFitness, 10);
        this.evolvedFitnessPlank.mesh.position.set(0, 0, 0);
        this.evolvedFitnessPlank.textMesh.position.set(0, 0, 0);
        this.evolvedFitnessPlank.move(Config.Simulation.Element.Position.EVOLVED_FITNESS_PLANK);
        this.evolvedFitnessPlank.rotate(Config.Simulation.Element.Angle.SIDE_VIEW);
        this.world.scene.add(this.evolvedFitnessPlank.mesh);
        this.world.scene.add(this.evolvedFitnessPlank.textMesh);


        this.timing = Config.Simulation.Timing.DESTROY_FRAMSTICK;
        if (this.evolutionMethod == State.MUTATION_GENOTYPE_SELECTION) {
          this.state = State.MUTATION_RETURN;
        } else if (this.evolutionMethod == State.CROSSOVER_GENOTYPE_SELECTION) {
          this.state = State.CROSSOVER_RETURN;
        } else if (this.evolutionMethod == State.CLONING_GENOTYPE_SELECTION) {
          this.state = State.CLONING_RETURN;
        }
        break;
	case State.CHANGE_GENOTYPE:
		this.genotypeEncoding = this.proposedGenotypeEncoding;
		this.world.setGenotypes(this.createGenotypes(Config.Simulation.NUMBER_OF_GENOTYPES,Config.Simulation.INITIAL_MUTATIONS),this.genotypeEncoding);
		this.proposedGenotypeEncoding = undefined;

		this.state = State.INITIAL;
		break;
    }
	if(this.isGenotypeChanging()){
		this.timing = 0;
	}
    this.lastTimeout = setTimeout(this.run.bind(this), this.timing);
  }

  setProposedGenotype(proposed){
	this.proposedGenotypeEncoding = proposed;
	if (this.lastTimeout){
		clearTimeout(this.lastTimeout);
		this.run();
	}
  }

  chooseIdx(){
	return this.weightedChoice();
  }

  randomChoice(){
	return parseInt(Math.random() * Config.Simulation.NUMBER_OF_GENOTYPES);
  }

  weightedChoice(){
	const fitness = this.world.table.fitnessPlanks
					.map((k)=>{
						return parseFloat(k.fitness)+0.0001;
					});
	const fitness_sum = fitness.reduce((partialSum, a) => partialSum + a, 0);
	const probs = fitness
					.map((k)=>{
						return k/fitness_sum;
					});

	var prob = Math.random();
	for(let i=0;i<probs.length;i++){
		if(prob < probs[i]){
			return i;
		}
		prob -= probs[i];
	}
	return -1;
  }

  createGenotypes(n,mutation_count){
	var func = this.getOperator(this.genotypeEncoding);
    let genoOper = new func();
    let genoOperatorsHelper = new Module.GenoOperatorsHelper(genoOper);
	var genotypes = [];

	for(let i=0;i<n;i++){
		var genotype = genoOperatorsHelper.getSimplest();
		genotypes.push(this.mutate(genotype,genoOperatorsHelper,mutation_count));
	}

	return genotypes;
  }

  getOperator(genoName){
	const genoType = genoName.replace("/*","").replace("*/","");
	var func = Module[`Geno_f${genoType}`];
	func = func?func:Module[`GenoOper_f${genoType}`];
	return func;
  }

  getMutatedGenotype(genotype){
	var func = this.getOperator(this.genotypeEncoding);
    let genoOper = new func();
    let genoOperatorsHelper = new Module.GenoOperatorsHelper(genoOper);
	genotype = this.mutate(genotype,genoOperatorsHelper,1);
	Module.destroy(genoOper);
    Module.destroy(genoOperatorsHelper);
    return genotype;
  }

  mutate(genotype,genoOperatorsHelper,times) {
	for(let i = 0;i<times;i++){
		genotype = this.tryMutation(genotype,genoOperatorsHelper);
	}
	return genotype;
  }

  tryMutation(genotype,genoOperatorsHelper){
	let genetics = new Module.PreconfiguredGenetics();
    let iteration = 0;
	do {
		if (genoOperatorsHelper.mutate(genotype.replace(this.genotypeEncoding, "")) == 0) {
		  let mutated = genoOperatorsHelper.getLastMutateGeno().c_str();
	  let newGenotype = `${this.genotypeEncoding}` + mutated;
  
		  let stringObj = new Module.SString();
		  stringObj.set(newGenotype);
		  let genoObj = new Module.Geno(stringObj);
  
		  if (genoObj.isValid()) {
			genotype = newGenotype;
			iteration = Config.Simulation.EVOLUTION_MAX_REPETITIONS;
		  }
  
		  Module.destroy(stringObj);
		  Module.destroy(genoObj);
		} else {
		  iteration = Config.Simulation.EVOLUTION_MAX_REPETITIONS;
		}
		iteration += 1;
	  } while(iteration < Config.Simulation.EVOLUTION_MAX_REPETITIONS);

	  return genotype;
  }

  getCrossoveredGenotype(genotype1, genotype2) {
	var func = this.getOperator(this.genotypeEncoding);
    let genoOper = new func();
    let genoOperatorsHelper = new Module.GenoOperatorsHelper(genoOper);
    let genetics = new Module.PreconfiguredGenetics();
    let iteration = 0;

    do {
      if (genoOperatorsHelper.crossOver(genotype1.replace(this.genotypeEncoding, ""),
                                        genotype2.replace(this.genotypeEncoding, "")) == 0) {
        let crossovered = genoOperatorsHelper.getLastCrossGeno1().c_str();
        let newGenotype = `${this.genotypeEncoding}` + crossovered;


        let stringObj = new Module.SString();
        stringObj.set(newGenotype);
        let genoObj = new Module.Geno(stringObj);

        if (genoObj.isValid()) {
          genotype1 = newGenotype;
          iteration = Config.Simulation.EVOLUTION_MAX_REPETITIONS;
        }

        Module.destroy(stringObj);
        Module.destroy(genoObj);
      } else {
        iteration = Config.Simulation.EVOLUTION_MAX_REPETITIONS;
      }
      iteration += 1;
    } while(iteration < Config.Simulation.EVOLUTION_MAX_REPETITIONS);

    Module.destroy(genoOper);
    Module.destroy(genoOperatorsHelper);
    return genotype1;
  }

  showEvolutionResult() {
    this.evolvedGenotypePlank = new GenotypePlank(this.evolvedGenotype, this.genotypeEncoding, 10);
    this.evolvedGenotypePlank.move(Config.Simulation.Element.Position.EVOLVED_GENOTYPE_PLANK);
    this.evolvedGenotypePlank.rotate(Config.Simulation.Element.Angle.SIDE_VIEW);

	this.neuroViewer.move(Config.Simulation.Element.Position.NEURAL_NETWORK);
    this.neuroViewer.rotate(Config.Simulation.Element.Angle.SIDE_VIEW);
    this.world.scene.add(this.evolvedGenotypePlank.mesh);
    this.world.scene.add(this.evolvedGenotypePlank.textMesh);
	this.world.scene.add(this.neuroViewer.get3DContainer());

    if (this.evolutionMethod == State.MUTATION_GENOTYPE_SELECTION) {
      this.showText(this.mutatedText);
      this.moveText(this.mutatedText, Config.Simulation.Element.Position.RESULT_TEXT);
      this.rotateText(this.mutatedText, Config.Simulation.Element.Angle.SIDE_VIEW);
    } else if (this.evolutionMethod == State.CROSSOVER_GENOTYPE_SELECTION) {
      this.showText(this.offspringText);
      this.moveText(this.offspringText, Config.Simulation.Element.Position.RESULT_TEXT);
      this.rotateText(this.offspringText, Config.Simulation.Element.Angle.SIDE_VIEW);
    } else if (this.evolutionMethod == State.CLONING_GENOTYPE_SELECTION) {
      this.showText(this.clonedText);
      this.moveText(this.clonedText, Config.Simulation.Element.Position.RESULT_TEXT);
      this.rotateText(this.clonedText, Config.Simulation.Element.Angle.SIDE_VIEW);
    }
  }

  evolutionReturn() {
    let replaceIdx = parseInt(Math.random() * Config.Simulation.NUMBER_OF_GENOTYPES);
    let genotypePlank = this.world.table.genotypePlanks[replaceIdx];
    let fitnessPlank = this.world.table.fitnessPlanks[replaceIdx];

    this.world.scene.remove(genotypePlank.mesh);
    this.world.scene.remove(genotypePlank.textMesh);
    this.world.scene.remove(fitnessPlank.mesh);
    this.world.scene.remove(fitnessPlank.textMesh);

    this.evolvedGenotypePlank.position = genotypePlank.position;
    this.evolvedGenotypePlank.move(this.evolvedGenotypePlank.position);
    this.evolvedGenotypePlank.rotate(Config.Simulation.Element.Angle.NORMAL);

    this.evolvedFitnessPlank.position = fitnessPlank.position;
    this.evolvedFitnessPlank.move(this.evolvedFitnessPlank.position);
    this.evolvedFitnessPlank.rotate(Config.Simulation.Element.Angle.NORMAL);

    this.world.table.genotypes[replaceIdx] = this.evolvedGenotype;
    this.world.table.genotypePlanks[replaceIdx] = this.evolvedGenotypePlank;
    this.world.table.fitnessPlanks[replaceIdx] = this.evolvedFitnessPlank;
	this.evolvedFitnessPlank = undefined;
    this.hideText(this.fitnessText);
	this.deactivateButtons();
  }

  getText(text, position) {
    let textMesh = Text.getInfoMesh(text);
    textMesh.position.set(position.x, position.y, position.z);
    textMesh.scale.set(0.01, 0.01, 0.01);
    return textMesh;
  }

  showText(text) {
    let tween = new TWEEN.Tween(text.scale).to({x: 1, y: 1, z: 1}, 500);
    tween.easing(TWEEN.Easing.Quadratic.InOut);
    tween.start();
  }

  hideText(text) {
    let tween = new TWEEN.Tween(text.scale).to({x: 0.01, y: 0.01, z: 0.01}, 500);
    tween.easing(TWEEN.Easing.Quadratic.InOut);
    tween.start();
  }

  getFramstick(genotype) {
    this.moveCamera(Config.Simulation.Camera.Position.FRAMSTICK);
    return new Framstick(genotype).mesh;
  }

  getArrow(idx) {
    return new Arrow(50 + (Config.Table.FitnessPlank.HEIGHT + Config.Table.Board.SPACING) * idx);
  }

  moveCamera(position) {
    let tween = new TWEEN.Tween(this.world.camera.position).to(position, Config.Simulation.Camera.SPEED);
    tween.easing(TWEEN.Easing.Quadratic.Out);
    tween.start();
  }

  moveText(textMesh, position) {
    let tween = new TWEEN.Tween(textMesh.position).to(position, Config.Simulation.Camera.SPEED);
    tween.easing(TWEEN.Easing.Quadratic.Out);
    tween.start();
  }

  rotateText(textMesh, angle) {
    let tween = new TWEEN.Tween(textMesh.rotation).to(angle, Config.Simulation.Camera.SPEED);
    tween.easing(TWEEN.Easing.Quadratic.Out);
    tween.start();
  }
}

var State = {
  "INITIAL": 0, "EVOLUTION_METHOD_SELECTION": 1, "MUTATION_GENOTYPE_SELECTION": 2,
  "MUTATION_SIDE_VIEW": 3, "MUTATION_RESULT": 4, "SHOW_FRAMSTICK": 5,
  "SHOW_RESULT": 6, "MUTATION_RETURN": 7, "CROSSOVER_GENOTYPE_SELECTION": 8,
  "CROSSOVER_SIDE_VIEW": 9, "CROSSOVER_RESULT": 10, "CROSSOVER_RETURN": 11,
  "DESTROY_FRAMSTICK": 12, "CLONING_GENOTYPE_SELECTION": 13,
  "CLONING_SIDE_VIEW": 14, "CLONING_RESULT": 15, "CLONING_RETURN": 16,
  "CHANGE_GENOTYPE": 17
};

var EVOLUTION_METHOD_TEXT_POSITION = {
  x: -210,
  y: 80 + (Config.Table.FitnessPlank.HEIGHT + Config.Table.Board.SPACING) * (Config.Simulation.NUMBER_OF_GENOTYPES + 1),
  z: -500
};
