/**
 * @file Table Object
 * @author Patryk Gliszczynski
 * @version 1.0
 */

class Table {
  /**
   * Class reposnsible for the visualization of the genotype and fitness table.
   */

  constructor(genotypes,genotypeType) {
    this.genotypes = genotypes;
	this.genotypeType = genotypeType;

    this.board = this.Board();
    this.genotypePlanks = this.GenotypePlanks();
    this.fitnessPlanks = this.FitnessPlanks();
    this.genotypeHeader = this.GenotypeHeader();
    this.fitnessHeader = this.FitnessHeader();
  }

  Board() {
    let texture = new THREE.TextureLoader().load(Config.Table.Board.TEXTURE);
    texture.wrapS = texture.wrapT = THREE.RepeatWrapping;
    texture.repeat.set(Config.Table.Board.REPEAT, Config.Table.Board.REPEAT);
    texture.anisotropy = Config.Table.Board.ANISOTROPY;
	texture.colorSpace = THREE.SRGBColorSpace;

    let geometry = new THREE.BoxGeometry(Config.Table.Board.WIDTH,
                                         (Config.Table.GenotypePlank.HEIGHT + Config.Table.Board.SPACING) * (this.genotypes.length + 2),
                                         Config.Table.Board.THICKNESS);

    let material = new THREE.MeshLambertMaterial({map: texture});
    material.opacity = 0;
    // material.blending = THREE.NoBlending;

    let board = new THREE.Mesh(geometry, material);
    board.position.set(Config.Table.Board.X_POS,
                       (Config.Table.GenotypePlank.HEIGHT + Config.Table.Board.SPACING) * (this.genotypes.length + 1) / 2,
                       Config.Table.Board.Z_POS)
    board.castShadow = true;
    board.receiveShadow = true;
    return board
  }

  GenotypePlanks() {
    let planks = []
    for (let i = 0; i < this.genotypes.length; i++) {
      planks.push(new GenotypePlank(this.genotypes[i],this.genotypeType, i));
    }
    return planks
  }

  FitnessPlanks() {
    let planks = []
    for (let i = 0; i < this.genotypes.length; i++) {
      planks.push(new FitnessPlank((Math.random() * 100).toFixed(2), i))
    }
    return planks;
  }

  GenotypeHeader() {
    return new GenotypePlank("Genotype",this.genotypeType, this.genotypes.length, true)
  }

  FitnessHeader() {
    return new FitnessPlank("Fitness", this.genotypes.length)
  }
}


class GenotypePlank {

  constructor(genotype, genotypeEncoding, position, bold) {
    this.genotype = genotype;
    this.position = {
      x: Config.Table.GenotypePlank.X_POS,
      y: 50 + (Config.Table.GenotypePlank.HEIGHT + Config.Table.Board.SPACING) * position,
      z: Config.Table.GenotypePlank.Z_POS
    };

    let texture = new THREE.TextureLoader().load(Config.Table.GenotypePlank.TEXTURE);
    texture.wrapS = texture.wrapT = THREE.RepeatWrapping;
    texture.repeat.set(Config.Table.GenotypePlank.REPEAT, Config.Table.GenotypePlank.REPEAT);
    texture.anisotropy = Config.Table.GenotypePlank.ANISOTROPY;
	texture.colorSpace = THREE.SRGBColorSpace;

    let geometry = new THREE.BoxGeometry(Config.Table.GenotypePlank.WIDTH,
                                         Config.Table.GenotypePlank.HEIGHT,
                                         Config.Table.GenotypePlank.THICKNESS);
    let material = new THREE.MeshLambertMaterial({map: texture});
    material.blending  = THREE.NoBlending;
    this.mesh = new THREE.Mesh(geometry, material);
    this.mesh.position.set(this.position.x, this.position.y, this.position.z);
    this.mesh.castShadow = Config.Table.GenotypePlank.CAST_SHADOW;
    this.mesh.receiveShadow = Config.Table.GenotypePlank.RECEIVE_SHADOW;

	this.genotypeType = genotypeEncoding;
    this.textMesh = Text.getGenotypeMesh(this.genotype,this.genotypeType);
    if (bold == true) {
      this.textMesh = Text.getFitnessMesh(this.genotype.replace(`${this.genotypeType} `, ""));
    }
    this.textMesh.position.set(this.position.x, this.position.y, this.position.z + 1);
  }

  updateGenotype(genotype,genotypeType){
	this.genotype = genotype;
	this.genotypeType = genotypeType;
	this.textMesh = Text.getGenotypeMesh(this.genotype,this.genotypeType);
	this.textMesh.position.set(this.position.x, this.position.y, this.position.z + 1);
  }

  move(position) {
    let tweenPlank = new TWEEN.Tween(this.mesh.position).to(position, Config.Table.GenotypePlank.Tween.Move.SPEED);
    tweenPlank.easing(Config.Table.GenotypePlank.Tween.Move.EASING);
    tweenPlank.start();

    let tweenText = new TWEEN.Tween(this.textMesh.position).to(position, Config.Table.GenotypePlank.Tween.Move.SPEED);
    tweenText.easing(Config.Table.GenotypePlank.Tween.Move.EASING);
    tweenText.start();
  }

  rotate(angle) {
    let tweenPlank = new TWEEN.Tween(this.mesh.rotation).to(angle, Config.Table.GenotypePlank.Tween.Rotate.SPEED);
    tweenPlank.easing(Config.Table.GenotypePlank.Tween.Rotate.EASING);
    tweenPlank.start();

    let tweenText = new TWEEN.Tween(this.textMesh.rotation).to(angle, Config.Table.GenotypePlank.Tween.Rotate.SPEED);
    tweenText.easing(Config.Table.GenotypePlank.Tween.Rotate.EASING);
    tweenText.start();
  }
}


class FitnessPlank {

  constructor(fitness, position) {
    this.fitness = fitness;
    this.position = {
      x: Config.Table.FitnessPlank.X_POS,
      y: 50 + (Config.Table.FitnessPlank.HEIGHT + Config.Table.Board.SPACING) * position,
      z: Config.Table.FitnessPlank.Z_POS
    };

    let texture = new THREE.TextureLoader().load(Config.Table.FitnessPlank.TEXTURE);
    texture.wrapS = texture.wrapT = THREE.RepeatWrapping;
    texture.repeat.set(Config.Table.FitnessPlank.REPEAT, Config.Table.FitnessPlank.REPEAT);
    texture.anisotropy = Config.Table.FitnessPlank.ANISOTROPY;
	texture.colorSpace = THREE.SRGBColorSpace;

    let geometry = new THREE.BoxGeometry(Config.Table.FitnessPlank.WIDTH,
                                         Config.Table.FitnessPlank.HEIGHT,
                                         Config.Table.FitnessPlank.THICKNESS);
    let material = new THREE.MeshLambertMaterial({map: texture});
    material.blending  = THREE.NoBlending;
    this.mesh = new THREE.Mesh(geometry, material);
    this.mesh.position.set(this.position.x, this.position.y, this.position.z);
    this.mesh.castShadow = Config.Table.FitnessPlank.CAST_SHADOW;
    this.mesh.receiveShadow = Config.Table.FitnessPlank.RECEIVE_SHADOW;

    this.textMesh = Text.getFitnessMesh(this.fitness);
    this.textMesh.position.set(this.position.x, this.position.y, this.position.z + 1);
  }

  updateFitness(value){
	this.fitness = value;
	this.textMesh.element.innerHTML = roundNumber(this.fitness,2);
  }

  move(position) {
    let tweenPlank = new TWEEN.Tween(this.mesh.position).to(position, Config.Table.FitnessPlank.Tween.Move.SPEED);
    tweenPlank.easing(Config.Table.FitnessPlank.Tween.Move.EASING);
    tweenPlank.start();

    let tweenText = new TWEEN.Tween(this.textMesh.position).to(position, Config.Table.FitnessPlank.Tween.Move.SPEED);
    tweenText.easing(Config.Table.FitnessPlank.Tween.Move.EASING);
    tweenText.start();
  }

  rotate(angle) {
    let tweenPlank = new TWEEN.Tween(this.mesh.rotation).to(angle, Config.Table.FitnessPlank.Tween.Rotate.SPEED);
    tweenPlank.easing(Config.Table.FitnessPlank.Tween.Rotate.EASING);
    tweenPlank.start();

    let tweenText = new TWEEN.Tween(this.textMesh.rotation).to(angle, Config.Table.FitnessPlank.Tween.Rotate.SPEED);
    tweenText.easing(Config.Table.FitnessPlank.Tween.Rotate.EASING);
    tweenText.start();
  }
}
