/**
 * @file Main Module
 * @author Patryk Gliszczynski
 * @version 1.0
 */

class App {
  /**
   * Main module responsible for the maintanance of the core animation components.
   */

  constructor() {
    this.scene = this.Scene();
	this.camera = this.Camera();
    this.webGLRenderer = this.WebGLRenderer();
	this.cameraControls = this.CameraControls();
    this.css3DRenderer = this.CSS3DRenderer();

	this.args = this.parseArguments();
	Config.Simulation.GENOTYPE = this.args["genotype_format"]?this.args["genotype_format"]:Config.Simulation.GENOTYPE;
    this.simulation = new Simulation(new World(this.camera, this.scene),this.args);

    document.body.appendChild(this.webGLRenderer.domElement);
    document.body.appendChild(this.css3DRenderer.domElement);
    window.addEventListener("resize", this.onWindowResize.bind(this), false);

	this.Controls();
	this.Options();
	this.NeuroView();
	this.FitnessRater();
  }

  NeuroView(){
	var thisApp = this;
	window.addEventListener('DOMContentLoaded',()=>{
		var neuroViewer = NeuroViewerView( Config.NeuroViewer );
		thisApp.simulation.neuroViewer = neuroViewer;
		neuroViewer.make3D();
	})
  }

  Controls(){
	var thisApp = this;
	window.addEventListener('DOMContentLoaded', () => {
		const controls = document.getElementsByClassName('controls')[0];
		thisApp.mutation_slider = new Slider(
			controls,
			"Mutation",
			function(prob){
				thisApp.simulation.setMutationProb(parseFloat(prob));
			});
		thisApp.crossover_slider = new Slider(
			controls,
			"Crossover",
			function(prob){
				thisApp.simulation.setCrossoverProb(parseFloat(prob));
			});
		thisApp.simulation.mutationListeners.push(thisApp.mutation_slider);
		thisApp.simulation.crossoverListeners.push(thisApp.crossover_slider);
		thisApp.simulation.reupdateListeners();
		return this;
	});
  }

  Options(){
	const encodings = this.findEncodings();

	var thisApp = this;
	window.addEventListener('DOMContentLoaded', () => {
		const selector = document.getElementById("genotype-selector");
		for(let i=0;i<encodings.length;i++){
			new Option(selector,encodings[i]);
		}
		selector.onchange = function(){
			thisApp.simulation.setProposedGenotype(`/*${this.value}*/`);
			// let url = new URL(window.location.href);
			// let params = new URLSearchParams(url.search);
			// let paramName = "genotype_format"; 
			// let paramValue = `/*${this.value}*/`;

			// if(params.has(paramName)){
			// 	params.set(paramName,paramValue);
			// }else{
			// 	params.append(paramName,paramValue);
			// }
			// window.location.href = window.location.origin+'/?'+params.toString();
		}
	});
  }

  FitnessRater(){
	var sim = this.simulation;
	window.addEventListener('DOMContentLoaded', ()=>{
		const rater = document.getElementById("fitness-rater");
		const thumbUp = new Button(rater,"👍",()=>{
			sim.updateFitness(5);
		});
		const thumbDown = new Button(rater,"👎",()=>{
			sim.updateFitness(-5);
		})

		sim.buttons.push(thumbUp);
		sim.buttons.push(thumbDown);
	});
  }

  findEncodings(){
	const functionName = "GenoConv_f";
	return Object
	.keys(Module) //framsticks module
	.filter((k)=>{return k.startsWith(functionName);})
	.map((k)=>{return k.replace(functionName,"")[0]});
  }

  Scene() {
    /**
     * Creates a new Scene object.
     */
    let scene = new THREE.Scene();
    scene.fog = new THREE.Fog(Config.Fog.BACKGROUND, Config.Fog.NEAR, Config.Fog.FAR);
    return scene;
  }

  Camera() {
    /**
     * Creates a new Camera object.
     */
    let camera = new THREE.PerspectiveCamera(Config.Camera.FOV, Config.Camera.ASPECT, Config.Camera.NEAR, Config.Camera.FAR);
    camera.position.set(Config.Camera.X_POS, Config.Camera.Y_POS, Config.Camera.Z_POS);
    camera.updateProjectionMatrix();
    return camera;
  }

  CameraControls() {
    /**
     * Creates a new Control object alowing to steer the camera.
     */
    let controls = new OrbitControls(this.camera,this.webGLRenderer.domElement );
    controls.autoRotate = Config.Controls.AUTO_ROTATE;
    controls.autoRotateSpeed = Config.Controls.AUTO_ROTATE_SPEED;
    controls.minDistance = Config.Controls.MIN_DISTANCE;
    controls.maxDistance = Config.Controls.MAX_DISTANCE;
    controls.maxPolarAngle = Config.Controls.MAX_POLAR_ANGLE;
	controls.enableKeys = false;
    controls.update();
    return controls
  }

  WebGLRenderer() {
    /**
     * Creates a WebGL Renderer object used to visualize 3D components.
     */
    let renderer = new THREE.WebGLRenderer({antialias: Config.WebGLRenderer.ANTIALIAS});
    renderer.setSize(Config.WebGLRenderer.WIDTH, Config.WebGLRenderer.HEIGHT);
    renderer.setPixelRatio(Config.WebGLRenderer.PIXEL_RATIO);
    renderer.shadowMap.enabled = Config.WebGLRenderer.ShadowMap.ENABLED;
    renderer.shadowMap.type = THREE.PCFSoftShadowMap;
    renderer.shadowMap.renderSingleSided = Config.WebGLRenderer.ShadowMap.SINGLE_SIDED;
    renderer.gammaInput = Config.WebGLRenderer.GAMMA_INPUT;
    renderer.gammaOutput = Config.WebGLRenderer.GAMMA_OUTPUT;
    return renderer;
  }

  CSS3DRenderer() {
    /**
     * Creates a CSS3D Renderer object used to visualize HTML+CSS components in 3D scene.
     */
    let renderer = new CSS3DRenderer();
    renderer.setSize(Config.CSS3DRenderer.WIDTH, Config.CSS3DRenderer.HEIGHT);
    renderer.domElement.style.position = Config.CSS3DRenderer.POSITION;
    renderer.domElement.style.top = Config.CSS3DRenderer.TOP;
	renderer.domElement.style.setProperty('pointer-events','none','');
    return renderer
  }

  run() {
    /**
     * Starts the simulation and rendering process.
     */
    this.render();
    this.simulation.run();
  }

  render() {
    /**
     * Main renderer loop, called on each frame.
     */
    requestAnimationFrame(this.render.bind(this));
    TWEEN.update();
    this.cameraControls.update();
    this.webGLRenderer.render(this.scene, this.camera);
    this.css3DRenderer.render(this.scene, this.camera);
  }

  onWindowResize() {
    /**
     * Updates the dimmensions of renderers after window size changes.
     */
    this.camera.aspect = window.innerWidth / window.innerHeight;
    this.camera.updateProjectionMatrix();
    this.webGLRenderer.setSize(window.innerWidth, window.innerHeight);
    this.css3DRenderer.setSize(window.innerWidth, window.innerHeight);
  }

  parseArguments() {
	/**
	 * Parses URL arguemnts
	 */

	const queryString = window.location.search;
	const urlParams = new URLSearchParams(queryString);
	return {
		"mutation_prob" : parseFloat(urlParams.get("mutation_prob")),
		"crossover_prob" : parseFloat(urlParams.get("crossover_prob")),
		"genotype_format": urlParams.get("genotype_format")
	};
  }
}

new App().run();
