// This file is a part of Framsticks SDK.  http://www.framsticks.com/
// Copyright (C) 1999-2023  Maciej Komosinski and Szymon Ulatowski.
// See LICENSE.txt for details.

// Copyright (C) 1999,2000  Adam Rotaru-Varga (adam_rotaru@yahoo.com), GNU LGPL
// Copyright (C) since 2001 Maciej Komosinski
// 2018, Grzegorz Latosinski, added development checkpoints and support for new API for neuron types

#include "f4_conv.h"
#include <common/log.h>
#include "../genooperators.h" //for GENOPER_OK constant

#ifdef DMALLOC
#include <dmalloc.h>
#endif


GenoConv_f40::GenoConv_f40()
{
	name = "Developmental encoding";
	in_format = '4';
	out_format = '0';
	mapsupport = 1;
}


SString GenoConv_f40::convert(SString &in, MultiMap *map, bool using_checkpoints)
{
	f4_Model *model = new f4_Model();
	int res = model->buildFromF4(in, using_checkpoints);
	if (res != GENOPER_OK)
	{
		delete model;
		return SString();  // oops
	}
	if (NULL != map)
		// generate to-f0 conversion map
		model->getCurrentToF0Map(*map);
	SString out = model->getF0Geno().getGenes();
	delete model;

	/* quick debugging test - print an approximate f1 conversion of every genotype converted to f0:
	GenoConv_F41_TestOnly conv41;
	SString f1 = conv41.convert(in, NULL, false);
	printf("f1 = %s\n", f1.c_str());
	*/

	return out;
}


GenoConv_F41_TestOnly::GenoConv_F41_TestOnly()
{
	name = "Only for testing, approximate f4->f1 converter";
	// Why approximate? for example, f1 does not allow to continue after branching: X(X,X)X  <-- the last X
	// Some modifier genes are also not perfectly converted.
	// And neuron properties are ignored...
	in_format = '4';
	out_format = '1';
	mapsupport = 0;
}


SString GenoConv_F41_TestOnly::convert(SString &in, MultiMap *map, bool using_checkpoints)
{
	f4_Model *model = new f4_Model();
	int res = model->buildFromF4(in, using_checkpoints);
	if (res != GENOPER_OK)
	{
		delete model;
		return SString();  // oops
	}
	SString out;
	model->toF1Geno(out);
	delete model;
	return out;
}


f4_Model::f4_Model() : Model()
{
	cells = NULL;
}

f4_Model::~f4_Model()
{
	if (cells) delete cells;
}

int f4_Model::buildFromF4(SString &geno, bool using_checkpoints)
{
	error = GENOPER_OK;
	errorpos = -1;

	// transform geno from string to nodes
	f4_Node f4rootnode;
	int res = f4_process(geno.c_str(), &f4rootnode);
	if (res || (f4rootnode.childCount() != 1)) //consider any error fatal, preventing building a model
	{
		error = GENOPER_OPFAIL;
		errorpos = res;
		return error;
	}

	// build cells, and simulate
	if (cells) delete cells;
	cells = new f4_Cells(f4rootnode.child, false);
	if (cells->getErrorCode() != GENOPER_OK)
	{
		error = cells->getErrorCode();
		errorpos = cells->getErrorPos();
		//delete cells;
		return error;
	}

	cells->simulate();
	if (cells->getErrorCode() != GENOPER_OK)
	{
		error = cells->getErrorCode();
		errorpos = cells->getErrorPos();
		return error;
	}

	// reset recursive traverse flags
	for (int i = 0; i < cells->cell_count; i++)
		cells->C[i]->recurProcessedFlag = false;

	open(using_checkpoints); // begin model build

	// process every cell
	for (int i = 0; i < cells->cell_count; i++)
	{
		int res = buildModelRecur(cells->C[i]);
		if (res)
		{
			logPrintf("f4_Model", "buildFromF4", LOG_ERROR, "Error %d when building a Model", res);
			error = res;
			break;
		}
	}

	int res_close = close();
	if (res_close == 0) // invalid
	{
		logPrintf("f4_Model", "buildFromF4", LOG_ERROR, "Error %d when closing a Model", res_close);
		error = -10;
	}

	return error;
}


f4_Cell* f4_Model::getStick(f4_Cell *C)
{
	if (C->type == f4_Cell_type::CELL_STICK) return C;
	if (NULL != C->dadlink)
		return getStick(C->dadlink);
	// we have no more dadlinks, find any stick
	for (int i = 0; i < cells->cell_count; i++)
		if (cells->C[i]->type == f4_Cell_type::CELL_STICK)
			return cells->C[i];
	// none!
	logMessage("f4_Model", "getStick", LOG_ERROR, "Not a single stick");
	return NULL;
}


int f4_Model::buildModelRecur(f4_Cell *C)
{
	if (C->recurProcessedFlag)
		// already processed
		return 0;

	// mark it processed
	C->recurProcessedFlag = true;

	// make sure parent is a stick
	if (C->dadlink != NULL)
		if (C->dadlink->type != f4_Cell_type::CELL_STICK)
		{
			C->dadlink = getStick(C->dadlink);
		}

	// make sure its parent is processed first
	if (C->dadlink != NULL)
	{
		int res = buildModelRecur(C->dadlink);
		if (res) return res;
	}

	char tmpLine[100];
	MultiRange range = C->genoRange;

	if (C->type == f4_Cell_type::CELL_STICK)
	{
		int jj_p1_refno;  // save for later
		// first end is connected to dad, or new
		if (C->dadlink == NULL)
		{
			// new part object for firstend
			// coordinates are left to be computed by Model
			sprintf(tmpLine, "fr=%g,ing=%g,as=%g",
				/*1.0/C->P.mass,*/ C->P.friction, C->P.ingestion, C->P.assimilation
				//C->firstend.x, C->firstend.y, C->firstend.z
			);
			jj_p1_refno = addFromString(PartType, tmpLine, &range);
			if (jj_p1_refno < 0) return -1;
			this->checkpoint();
		}
		else {
			// adjust mass/vol of first endpoint
			jj_p1_refno = C->dadlink->p2_refno;
			Part *p1 = getPart(jj_p1_refno);
			p1->mass += 1.0;
			//      p1->volume += 1.0/C->P.mass;
		}
		// new part object for lastend
		sprintf(tmpLine, "fr=%g,ing=%g,as=%g",
			//C->lastend.x, C->lastend.y, C->lastend.z
			/*"vol=" 1.0/C->P.mass,*/ C->P.friction, C->P.ingestion, C->P.assimilation
		);
		C->p2_refno = addFromString(PartType, tmpLine, &range);
		if (C->p2_refno < 0) return -2;

		// new joint object
		// check that the part references are valid
		int jj_p2_refno = C->p2_refno;
		if ((jj_p1_refno < 0) || (jj_p1_refno >= getPartCount())) return -11;
		if ((jj_p2_refno < 0) || (jj_p2_refno >= getPartCount())) return -12;
		sprintf(tmpLine, "p1=%d,p2=%d,dx=%g,dy=0,dz=0,rx=%g,ry=0,rz=%g"\
			",stam=%g",
			jj_p1_refno, jj_p2_refno,
			// relative position -- always (len, 0, 0), along the stick
			// this is optional!
			C->P.length,
			// relative rotation
			C->xrot, C->zrot,
			//C->P.ruch,   // rotstif
			C->P.stamina
		);
		C->joint_refno = addFromString(JointType, tmpLine, &range);
		if (C->joint_refno < 0) return -13;
		this->checkpoint();
	}

	if (C->type == f4_Cell_type::CELL_NEURON)
	{
		const char* nclass = C->neuclass->name.c_str();
		if (C->neuclass->getPreferredLocation() == 0)
		{
			if (strcmp(nclass, "N") == 0) //special case just to specify the only neuron properties supported by f4, i.e., the properties for neuron class 'N'
				sprintf(tmpLine, "d=\"N:in=%g,fo=%g,si=%g\"", C->inertia, C->force, C->sigmo);
			else
				sprintf(tmpLine, "d=\"%s\"", nclass);
		}
		else if (C->neuclass->getPreferredLocation() == 1) // attached to Part or have no required attachment - also part
		{
			int partno = C->dadlink->p2_refno;
			if ((partno < 0) || (partno >= getPartCount())) return -21;

			sprintf(tmpLine, "p=%d,d=\"%s\"", partno, nclass);
		}
		else // attached to Joint, assume there are only three possibilities of getPreferredLocation()
		{
			int jointno = C->dadlink->joint_refno;

			if (strcmp(nclass, "@") == 0)
				sprintf(tmpLine, "j=%d,d=\"@:p=%g\"", jointno, C->P.muscle_power);
			else if (strcmp(nclass, "|") == 0)
			{
				sprintf(tmpLine, "j=%d,d=\"|:p=%g,r=%g\"", jointno, C->P.muscle_power, C->dadlink->P.muscle_bend_range); //Macko 2023-05 change: we take muscle_bend_range from dadlink, not from C, because we also assign this neuron to C->dadlink->joint_refno. Without this, for example in /*4*/<<X><<<<X>N:|>X>X>X>X the muscle is attached to the junction with 3 sticks, but gets range=33% as in a four-stick junction. f1 correctly sets range=0.5 (see also conv_f1_f0_branch_muscle_range) for the analogous phenotype: X(X[|],X(X,X,X))
			}
			else
				sprintf(tmpLine, "j=%d,d=\"%s\"", jointno, nclass);
		}

		C->neuro_refno = addFromString(NeuronType, tmpLine, &range);
		if (C->neuro_refno < 0) return -22;
		this->checkpoint();

		for (int j = 0; j < C->conns_count; j++)
		{
			if (C->conns[j]->from != NULL)
				buildModelRecur(C->conns[j]->from);

			tmpLine[0] = 0;
			if (C->conns[j]->from == NULL)
			{
				logMessage("f4_Model", "buildModelRec", LOG_ERROR, "Old code for sensors as inputs embedded in [connection]: C->conns[j]->from == NULL");
			}
			int from = -1;
			if (C->conns[j]->from != NULL) // input from another neuron
				from = C->conns[j]->from->neuro_refno;
			if (from >= 0)
			{
				sprintf(tmpLine, "%d,%d,%g", C->neuro_refno, from, C->conns[j]->weight);
				if (addFromString(NeuronConnectionType, tmpLine, &range) < 0) return -35;
				this->checkpoint();
			}
		}
	}
	return 0;
}


void f4_Model::toF1Geno(SString &out)
{
	cells->toF1Geno(out);
}
