// This file is a part of Framsticks SDK.  http://www.framsticks.com/
// Copyright (C) 1999-2023  Maciej Komosinski and Szymon Ulatowski.
// See LICENSE.txt for details.

// Copyright (C) 1999,2000  Adam Rotaru-Varga (adam_rotaru@yahoo.com), GNU LGPL

#ifndef _F4_GENERAL_H_
#define _F4_GENERAL_H_

#include <frams/util/3d.h>
#include <frams/util/sstring.h>
#include <frams/util/multirange.h>
#include <frams/genetics/geneprops.h>

#ifdef DMALLOC
#include <dmalloc.h>
#endif

/**
 * Performs single rotation angle decrementation on a given value.
 * @param v pointer to the decremented value
 */
void rolling_dec(double *v);

/**
 * Performs single rotation angle incrementation on a given value.
 * @param v pointer to the incremented value
 */
void rolling_inc(double *v);

class f4_Node;   // later
class f4_Cell;   // later
class f4_Cells;  // later

enum class f4_Cell_type {
	CELL_UNDIFF, ///<undifferentiated cell
	CELL_STICK,  ///<differentiated to stick, cannot divide
	CELL_NEURON  ///<differentiated to neuron, can divide
};

class f4_CellConn;

/** @name Constraints of f4 genotype structures */
//@{
#define F4_MAX_CELL_INPUTS  10 ///<maximum number of neuron inputs in a developing organism
#define F4_MAX_CELLS 100 ///<maximum number of f4 organism cells
//@}

/**
 * Abstract cell type - the representation of a single component in the developmental
 * encoding. In the beginning, each f4_Cell is undifferentiated. During the process
 * of development it can divide or differentiate into a stick or a neuron. If it
 * differentiates to a neuron, then it preserves the ability to divide, but divided
 * cells will be the same type as the parent cell. If it is a stick, then it cannot
 * be divided anymore.
 *
 * From f4_Cell array the final Model of a creature is created.
 */
class f4_Cell
{
public:
	/**
	 * Represents the repetition marker. It holds information about the pointer
	 * to the repetition node and the count of repetitions.
	 */
	class repeat_ptr
	{
	public:
		repeat_ptr() : node(NULL), count(-1) { };

		/**
		 * A constructor that takes the pointer to the repetition node and the count of repetitions.
		 * @param a pointer to f4_Node for repetition character
		 * @param b the number of repetitions
		 */
		repeat_ptr(f4_Node *a, int b) : node(a), count(b) { };

		inline void makeNull() { node = NULL; count = -1; };

		inline bool isNull() const { return ((node == NULL) || (count <= 0)); };

		inline void dec() { count--; };
		f4_Node    *node; ///<pointer to the repetition code
		int       count; ///<repetition counter
	};

	/**
	 * Represents the stack of repeat_ptr objects. The objects are
	 * pushed to the stack when '#' repetition symbol appears, and are popped when
	 * the end of the current cell definition, i.e. the '>' character, appears. After the
	 * '>' character, the cell is duplicated as many times as it is defined after the
	 * repetition marker.
	 */
	class repeat_stack
	{
	public:
		repeat_stack() { top = 0; }

		inline void clear() { top = 0; }

		/**
		 * Pushes repeat_ptr object onto the stack. If the stack size is exceeded, then no
		 * information is provided.
		 * @param rn repetition node info
		 */
		inline void push(repeat_ptr rn) { if (top >= stackSize) return; ptr[top] = rn; top++; }

		inline void pop() { if (top > 0) top--; }

		/**
		 * Gets the current top element.
		 * @return pointer to the element on top of the repeat_stack object
		 */
		inline repeat_ptr* first() { return &(ptr[top - (top > 0)]); };
		static const int stackSize = 4;  ///<max 4 nested levels
		repeat_ptr ptr[stackSize]; ///<array holding pointers to repeat_ptr
		int top;  ///<index of the top of the stack
	};

	/**
	 * Creates a new f4_Cell object.
	 * @param nnr number of the cell
	 * @param ndad pointer to the parent of the created cell
	 * @param nangle the amount of commas affecting branch angles
	 * @param newP genotype properties of a given cell
	 */
	f4_Cell(int nnr, f4_Cell *ndad, int nangle, GeneProps newP);
	/**
	 * Creates a new f4_Cell object.
	 * @param nO pointer to an organism containing the cell
	 * @param nnr number of the cell
	 * @param ngeno pointer to the root of the genotype tree
	 * @param ngcur pointer to the f4_Node representing the current cell in the genotype tree
	 * @param ndad pointer to the parent of the created cell
	 * @param nangle the number of commas affecting branch angles
	 * @param newP genotype properties of a given cell
	 */
	f4_Cell(f4_Cells *nO, int nnr, f4_Node *ngeno, f4_Node *ngcur, f4_Cell *ndad, int nangle, GeneProps newP);

	~f4_Cell();

	/**
	 * Performs a single step of cell development. This method requires a pointer to
	 * the f4_Cells object in org attribute. If the current node in genotype tree
	 * is the branching character '<', the cell divides into two cells, unless the
	 * cell was already differentiated into the stick cell. Otherwise, the current
	 * differentiation or modification is performed on the cell. If current node is
	 * creating a connection between two neuron nodes and the input node is not
	 * yet developed, the simulation of the development of the current cell returns
	 * to wait until the input node is created. The oneStep method is deployed for every cell
	 * at least once. If one cell requires another one to develop, oneStep
	 * should be deployed again on this cell.
	 *
	 * This method, unlike genotype tree creation, checks semantics. This means that
	 * this function will fail (set error code) if:
	 *  - the cell differentiated as a stick will have branching node '<',
	 *  - the undifferentiated cell will have termination node '>' (end of cell development without differentiation),
	 *  - the stack of repetition marker '#' will exceed maximum allowed value of repetition,
	 *  - the stick modifiers, like rotation, will be applied on neuron cell,
	 *  - the differentiated cell will be differentiated again,
	 *  - the connection between neurons cannot be established,
	 *  - the neuron class is not valid.
	 *
	 * This function returns either because the development of this cell was completed,
	 * or it was halted (yielding to other cells), or the error code was set in the f4_Cells object in the org attribute.
	 */
	void oneStep();

	/**
	 * Adds a connection between this neuron cell and a given neuron cell in nfrom.
	 * @param nfrom input neuron cell
	 * @param nweight weight of connection
	 * @return 0 if connection is established, -1 otherwise
	 */
	int   addConnection(f4_Cell *nfrom, double nweight);

	/**
	 * Adjusts properties of stick objects.
	 */
	void  adjustRecur();

	int        nr;                 ///<number of cell (seems to be used only in the approximate f1 converter for neuron connections)
	f4_Cell_type type;             ///<type
	f4_Cell *dadlink;              ///<pointer to cell parent
	f4_Cells  *org;	               ///<uplink to organism

	f4_Node *genot; 	           ///<genotype tree
	f4_Node *gcur;                 ///<current genotype execution pointer
	f4_Node *old_gcur;             ///<used externally by f4_Cells::oneStep() to track changes of gcur, i.e., to detect progress in cell development
	repeat_stack repeat;           ///<stack holding repetition nodes and counters
	bool recurProcessedFlag;       ///<used during recursive traverse
	MultiRange genoRange;          ///<remember the genotype codes affecting this cell so far

	GeneProps    P;                ///<properties
	int          anglepos;         ///<number of position within dad's children (,)
	int          stickchildcount;  ///<number of children (sticks only)
	int          commacount;       ///<number of postitions at lastend (>=childcount)
	double       rolling;          ///<rolling angle ('R') (around x)
	double       xrot;			   ///<rotation angle around x
	double       zrot;             ///<horizontal rotation angle due to branching (around z)

	int          p2_refno;         ///<the number of the last end part object, used in f0
	int          joint_refno;      ///<the number of the joint object, used in f0
	int          neuro_refno;      ///<the number of the neuro object, used in f0

	double       inertia;          ///<inertia of neuron N
	double       force;            ///<force of neuron N
	double       sigmo;            ///<sigmoid of neuron N
	f4_CellConn *conns[F4_MAX_CELL_INPUTS]; ///<array of neuron connections
	int          conns_count;      ///<number of connections
	NeuroClass *neuclass;          ///<pointer to neuron class
};

/**
 * Class representing a connection between neuron cells.
 */
class f4_CellConn
{
public:
	/**
	 * Constructor for f4_CellLink class. Parameter nfrom represents input
	 * neuron cell.
	 * @param nfrom pointer to input neuron cell
	 * @param nweight weight of connection
	 */
	f4_CellConn(f4_Cell *nfrom, double nweight);

	f4_Cell *from;  ///<pointer to input neuron cell
	double weight;  ///<weight of connection
};


/**
 * A class representing a collection of cells. It is equivalent to an organism.
 */
class f4_Cells
{
public:

	/**
	 * Constructor taking genotype in a form of a tree.
	 * @param genome genotype tree
	 * @param nrepair false if nothing to repair
	 */
	f4_Cells(f4_Node *genome, bool nrepair);

	/**
	 * Destructor removing cells from memory.
	 */
	~f4_Cells();

	/**
	 * Adds a new cell to organism.
	 * @param newcell cell to be added
	 */
	void addCell(f4_Cell *newcell);

	/**
	 * Creates an approximate genotype in the f1 encoding and stores it in a given parameter.
	 * @param out the string in which the approximate f1 genotype will be stored
	 */
	void toF1Geno(SString &out);

	/**
	 * Performs a single step of organism development. It runs each active cell in the organism.
	 * @return false if all cells are developed or there is an error, true otherwise
	 */
	bool oneStep();

	/**
	 * Performs the full development of organism and returns error code if something
	 * went wrong.
	 * @return 0 if organism developed successfully, error code if something went wrong
	 */
	int simulate();

	/**
	 * Prints the current state of the organism (for debugging purposes).
	 * @param description printout header
	 */
	void print_cells(const char* description);

	/**
	 * Returns error code of the last simulation.
	 * @return error code
	 */
	int getErrorCode() { return errorcode; };

	/**
	 * Returns position of an error in genotype.
	 * @return position of an error
	 */
	int getErrorPos() { return errorpos; };

	/**
	 * Sets error code GENOPER_OPFAIL for a simulation on a given position.
	 * @param nerrpos position of an error
	 */
	void setError(int nerrpos);

	/**
	 * Sets the element of genotype to be repaired by removal.
	 * @param nerrpos position of an error in genotype
	 * @param to_remove the f4_Node to be removed from the genotype tree in order to repair
	 */
	void setRepairRemove(int nerrpos, f4_Node *to_remove);

	/**
	 * Sets repairing of a genotype by inserting a new node to the current genotype.
	 * @param nerrpos position of an error in genotype
	 * @param parent the parent of a new element
	 * @param to_insert the element to be inserted
	 * @return 0 if repair can be performed, or -1 otherwise because the repair flag wasn't set in the constructor
	 */
	int setRepairInsert(int nerrpos, f4_Node *parent, f4_Node *to_insert);

	/**
	 * Repairs the genotype according to setRepairRemove or setRepairInsert methods.
	 * @param geno pointer to the genotype tree
	 * @param whichchild 1 if first child, 2 otherwise
	 */
	void repairGeno(f4_Node *geno, int whichchild);

	// the cells
	f4_Cell *C[F4_MAX_CELLS];  ///<Array of all cells of an organism
	int     cell_count;        ///<Number of cells in an organism
	bool    development_stagnation; ///< simulate() and oneStep() use it to force f4_Cell's waiting to develop their neural connections to progress, indicating that all cells have not had progress during the last step

private:
	// for error reporting / genotype fixing
	bool repair;
	int errorcode;
	int errorpos;
	f4_Node *repair_remove;
	f4_Node *repair_parent;
	f4_Node *repair_insert;
	void toF1GenoRec(int curc, SString &out);
	f4_Cell *tmpcel;  // needed by toF1Geno
};


/**
 * A class to organize a f4 genotype in a tree structure.
 */
class f4_Node
{
public:
	string name; ///<one-letter gene code or multiple characters for neuron classes (then neuclass != NULL)
	f4_Node *parent; ///<parent link or NULL
	f4_Node *child; ///<child or NULL
	f4_Node *child2; ///<second child or NULL
	int pos; ///<original position in the string

	int reps; ///<repetition counter for the '#' gene
	char prop_symbol; ///<old-style properties (force,intertia,sigmoid) of the N neuron: !=/
	bool prop_increase; ///<false=decrease neuron property (force,intertia,sigmoid), true=increase it
	int conn_from; ///<relative number of the neuron this neuron get an input from
	double conn_weight; ///<neuron connection weight
	NeuroClass *neuclass; ///< NULL or not if "name" is a neuroclass name with a proper genotype context ("N:neuroclassname"). New in 2023-04 - to fix fatal flaw with fundamental assumptions: it was impossible to distinguish between single-character neuron names such as S, D, G and single-character modifiers. They were all stored in the "name" field. Before 2018 this was never a problem because the only supported neuroclasses had distinctive symbols such as @|*GTS, and the set of supported modifiers was small and different from neuroclass letters (no G,D,S clash).

	f4_Node();

	/**
	 * Multiple-character name constructor.
	 * @param nname string from genotype representing node
	 * @param nparent pointer to parent of the node
	 * @param npos position of node substring in the genotype string
	 */
	f4_Node(string nname, f4_Node *nparent, int npos);

	/**
	 * Single-character name constructor.
	 * @param nname character from genotype representing node
	 * @param nparent pointer to parent of the node
	 * @param npos position of node character in the genotype string
	 */
	f4_Node(char nname, f4_Node *nparent, int npos);

	~f4_Node();

	/**
	 * Recursively print subtree (for debugging).
	 * @param root starting node
	 * @param indent initial indentation
	 */
	static void print_tree(const f4_Node *root, int indent);

	/**
	 * Adds the child to the node.
	 * @param nchi the child to be added to the node
	 * @return 0 if the child could be added, -1 otherwise
	 */
	int addChild(f4_Node *nchi);

	/**
	 * Removes the child from the node.
	 * @param nchi the child to be removed from the node
	 * @return 0 if child could be removed, -1 otherwise
	 */
	int removeChild(f4_Node *nchi);

	/**
	 * Returns the number of children.
	 * @return 0, 1 or 2
	 */
	int childCount();

	/**
	 * Returns the number of nodes coming from this node in a recursive way.
	 * @return the number of nodes from this node
	 */
	int count() const;

	/**
	 * Returns the nth subnode (0-)
	 * @param n index of the child to be found
	 * @return pointer to the nth subnode or NULL if not found
	 */
	f4_Node* ordNode(int n);

	/**
	 * Returns a random subnode.
	 * @return random subnode
	 */
	f4_Node* randomNode();

	/**
	 * Returns a random subnode with a given size.
	 * @param min minimum size
	 * @param max maximum size
	 * @return a random subnode with a given size or NULL
	 */
	f4_Node* randomNodeWithSize(int min, int max);

	/**
	 * Prints recursively the tree from a given node.
	 * @param buf variable to store printing result
	 */
	void      sprintAdj(char *&buf);

	/**
	 * Recursively copies the genotype tree from this node.
	 * @return pointer to a tree copy
	 */
	f4_Node* duplicate();

	/**
	 * Recursively releases memory from all node children.
	 */
	void      destroy();
private:
	void     sprint(SString &out);	// print recursively
};

/**
 * The main function for converting a string of f4 encoding to a tree structure. Prepares
 * f4_Node root of tree and runs f4_processRecur function for it.
 * @param geno the string representing an f4 genotype
 * @return a pointer to the f4_Node object representing the f4 tree root
 */
 //f4_Node* f4_processTree(const char *geno);

 /**
  * Scans a genotype string starting from a given position. This recursive method creates
  * a tree of f4_Node objects. This method extracts each potentially functional element
  * of a genotype string to a separate f4_Nodes. When the branching character '<' occurs,
  * f4_processRecur is deployed for the latest f4_Node element. This method does not
  * analyse the genotype semantically, it only checks if the syntax is proper. The only
  * semantic aspect is neuron class name extraction, where the GenoOperators
  * class is used to parse the potential neuron class name.
  * This is an internal function; for regular cases, use f4_process().
  * @param genot the string with the entire genotype
  * @param genot_len length of genot (precomputed for efficiency)
  * @param pos_inout the current position of processing in string (advanced by the function)
  * @param parent current parent of the analysed branch of the genotype
  * @return 0 if processing was successful, otherwise returns the position of an error in the genotype
  */
int f4_processRecur(const char *genot, const int genot_len, int &pos_inout, f4_Node *parent);

/**
 * A wrapper for f4_processRecur(). Creates a tree of f4_Node objects corresponding to
 * the provided genotype.
 * @param genot the string with the entire genotype
 * @param root root of the tree corresponding to the genotype
 * @return 0 if processing was successful, otherwise returns the position of an error in the genotype
 */
int f4_process(const char *genot, f4_Node *root);

/**
 * Parses notation of the neuron connection - takes the beginning of the connection
 * definition, extracts the relative position of input neurons and the weight of the connection.
 * After successful parsing, returns the pointer to the first character after the connection
 * definition, or NULL if the connection definition was not valid due to the lack of [, :, ]
 * characters or an invalid value of relfrom or weight.
 * @param fragm the beginning of connection definition, should be the '[' character
 * @param relfrom the reference to an int variable in which the relative position of the input neuron will be stored
 * @param weight the reference to a double variable in which the weight of the connection will be stored
 * @return the pointer to the first character in string after connection definition
 */
const char *parseConnection(const char *fragm, int &relfrom, double &weight);

#endif
