// This file is a part of Framsticks SDK.  http://www.framsticks.com/
// Copyright (C) 1999-2023  Maciej Komosinski and Szymon Ulatowski.
// See LICENSE.txt for details.

// Copyright (C) 1999,2000  Adam Rotaru-Varga (adam_rotaru@yahoo.com), GNU LGPL
// Copyright (C) since 2001 Maciej Komosinski

#ifndef _F4_OPER_H_
#define _F4_OPER_H_

#include <stdio.h>
#include "f4_general.h"
#include "common/nonstd.h"
#include "../genooperators.h"
#include <frams/param/param.h>

/** @name Codes for general mutation types */
//@{
#define F4_ADD           0 ///<Adding new node
#define F4_DEL           1 ///<Deletion of node
#define F4_MOD           2 ///<Modification of node
#define F4_COUNT         3 ///<Count of mutation types
//@}

/** @name Codes for specific F4_ADD mutation subtypes */
//@{
#define F4_ADD_DIV       0
#define F4_ADD_CONN      1
#define F4_ADD_NEUPAR    2
#define F4_ADD_REP       3
#define F4_ADD_SIMP      4
#define F4_ADD_COUNT     5
//@}

/** @name Codes for specific F4_MOD neuron mutation subtypes (not included in mutation_method_names, all are considered F4_MOD there) */
//@{
#define F4_MODNEU_CONN     0
#define F4_MODNEU_WEIGHT   1
#define F4_MODNEU_COUNT    2
//@}

class Geno_f4 : public GenoOperators
{
public:
	Geno_f4();
	void setDefaults();

	int checkValidity(const char *, const char *genoname);
	int validate(char *&, const char *genoname);
	int mutate(char *& g, float & chg, int &method);
	int crossOver(char *&g1, char *&g2, float& chg1, float& chg2);
	const char* getSimplest() { return "X"; }
	uint32_t style(const char *g, int pos);

	// mutation probabilities
	double prob[F4_COUNT];            ///<relative probabilities of selecting mutation types in f4 genotype
	double probadd[F4_ADD_COUNT];     ///<relative probabilities of selecting addition mutation subtypes
	double probmodneu[F4_MODNEU_COUNT];     ///<relative probabilities of selecting neuron mutation subtypes
	paInt mut_max_rep;                ///maximum allowed number of repetitions for the '#' repetition gene

	SString excluded_modifiers;       ///<Modifiers that are excluded in mutation process
	static const char *all_modifiers;

private:

	/**
	 * Validates a f4 genotype. If the genotype is invalid, the genotype is repaired
	 * and the validation is repeated. Validation is performed as long as repairing
	 * is not effective, or the number of retries exceeded the given limit.
	 * @param geno genotype tree
	 * @param retrycount maximum amount of repair retries
	 * @return GENOOPER_OK if genotype is valid, GENOPER_REPAIR if genotype can be repaired, GENOPER_OPFAIL if genotype can't be repaired
	 */
	int  ValidateRecur(f4_Node *geno, int retrycount) const;

	/**
	 * Performs mutation of an f4 genotype. The mutation is performed on a random node
	 * from a given tree. The method of mutation is chosen via the roulette selection,
	 * where probabilities of choosing each mutation type are given in the 'prob'
	 * array. Possible mutation types are:
	 *  - F4_ADD - adds new element to the genotype by:
	 *   - F4_ADD_DIV - replacing randomly selected node with division node '<', setting this node as a child and creating new stick or neuron sibling of the selected cell (the neuron-type sibling will be connected to a random existing neuron),
	 *   - F4_ADD_CONN - adding connection for an existing neuron,
	 *   - F4_ADD_NEUPAR - adding neuron property to the selected node, if it is a neuron node,
	 *   - F4_ADD_REP - adding a repetition node before a randomly selected node (the repetition node has 2 repetitions),
	 *   - F4_ADD_SIMP - adding a simple node before a selected node,
	 *  - F4_DEL - removes a randomly selected node (the node needs to have a parent and at least one child, otherwise returns GENOPER_OPFAIL),
	 *  - F4_MOD - modifies one of simple nodes by:
	 *   - '<' - swapping children in division
	 *   - '[' - modifying connection of a neuron
	 *   - '#' - incrementing or decrementing repetition count
	 *
	 * @param g input genotype; the result of mutation will be stored in this parameter
	 * @param method reference to the variable that will get the selected method of mutation
	 * @return GENOPER_OK if mutation was performed successfully, GENOPER_FAIL otherwise
	 */
	int  MutateOne(f4_Node *& g, int &method) const;

	/**
	 * Finds all neurons in g (in the order of ordNode()) and returns their neuroclasses in a vector.
	 * Additionally, looks for the needle_neuron node and returns its index (in the list of the returned vector) as found_index,
	 * or -1 if not found (for example, it was not a neuroclass node or not added to the "g" tree).
	 * @param g root node
	 * @param needle_neuron neuroclass node to look for in all nodes
	 * @param found_index returned index of needle
	 * @return all nodes that are neurons
	 */
	static vector<NeuroClass*> findAllNeuronsAndNode(f4_Node * const & g, f4_Node* const &needle_neuron, int &found_index);

	/**
	 * Finds indexes of a given neuron and another random (output- or input-providing) neuron in the list of all neurons present in the "g" tree.
	 * @param g root node
	 * @param neuron neuroclass node to look for in all nodes in g
	 * @param other_has_output if true, other neuron will provide output; otherwise, it will accept input(s)
	 * @param neuron_index returned index of neuron
	 * @param other_neuron_index returned index of a random neuron that provides an output or accepts inputs
	 * @return true if succeeded, false otherwise
	 */
	static bool findConnectionNeuronIndexes(f4_Node * const &g, f4_Node *neuron, bool other_has_output, int &neuron_index, int &other_neuron_index);

	/**
	 * Creates a random connection to an existing neuron and randomizes connection weight
	 * sensor for a neuron.
	 * @param nn neuron class node
	 * @param nn_index index of the nn neuron
	 * @param other_index index of the neuron providing output, to get input from
	 */
	void connectionNodeChangeRandom(f4_Node *nn, int nn_index, int other_index) const;

	/**
	 * Introduces a random modification to the neuron node.
	 * @param nn neuron node
	 */
	void nparNodeMakeRandom(f4_Node *nn) const;

	/**
	 * Increases or decreases the amount of repetitions in the repetition node.
	 * @param nn repetition node
	 */
	void repeatNodeChangeRandom(f4_Node *nn) const;

	/**
	 * Tries to perform a mutation until success. There is a maximum of 20 tries. Returns GENOPER_OK or GENOPER_OPFAIL.
	 * @param g genotype tree
	 * @param method reference to the variable that will get the selected method of mutation
	 * @return GENOPER_OK if performed successful mutation, GENOPER_FAIL otherwise
	 */
	int  MutateOneValid(f4_Node *&g, int &method) const;

	/**
	 * Performs crossover of two creatures. The 'chg' parameter determines approximately what
	 * percentage of the first creature should form the offspring. '1-chg' is the percentage
	 * of the second creature in the offspring.
	 * @param g1 first parent
	 * @param g2 second parent
	 * @param chg percentage of the first parent in offspring (the second parent has the rest)
	 */
	int  CrossOverOne(f4_Node *g1, f4_Node *g2, float chg) const;
};


#endif

