// This file is a part of Framsticks SDK.  http://www.framsticks.com/
// Copyright (C) 1999-2020  Maciej Komosinski and Szymon Ulatowski.
// See LICENSE.txt for details.

#ifndef _GREEDY_MEASURE_H
#define _GREEDY_MEASURE_H

#include "measure-mds-based.h"
#include "frams/genetics/geno.h"
#include "frams/model/model.h"
#include "simil-match.h"
#include "measure-greedy.h"

#define TDN_SIZE 5


enum TDNELEMS2
{
	ORG_IND = 0,
	DEG = 1,
	NEUR_CONNS = 2,
	NEUROS = 3,
	FUZ_DEG = 4
};

class SimilMeasureGreedy : public SimilMeasureMDSBased
{
public:
	SimilMeasureGreedy();
	~SimilMeasureGreedy(){};

	static int compareDegrees(const void *pElem1, const void *pElem2);
	static int compareFuzzyDegrees(const void *pElem1, const void *pElem2);
	static int compareConnsNo(const void *pElem1, const void *pElem2);
	static int getNOFactors();
	int setParams(std::vector<double> params);
	/// Interface to local parameters
	Param localpar;
	/// Table of weights for weighted distance function.
	/// Weights are for factors in the following order:
	/// [0]: m_iDV (difference in the number of vertices)
	/// [1]: m_iDD (difference in degrees over matching)
	/// [2]: m_iDN (difference in neurons over matching)
	/// [3]: m_dDG (difference in geometry over matching)
	/// @sa EvaluateDistance
	double m_adFactors[4];

	//Controls the depth of fuzzy neighbourhood 
	int fuzzyDepth;
	bool isFuzzy;

protected:
	double distanceForTransformation();
	double distanceWithoutAlignment();
	void prepareData();
	void beforeTransformation();
	void computeMatching();
	void copyMatching();
	void cleanData();

	int sortPartInfoTables();
	int countPartNeurons();
	int countPartDegrees();
	int countPartsDistance();

	void countFuzzyNeighb();
	void sortFuzzyNeighb();
	void getNeighbIndexes(int mod, int partInd, std::vector<int> &indexes);
	void fuzzyOrder();
	int createPartInfoTables();
	
	void _PrintArray(int *array, int base, int size);
	void _PrintArrayDouble(double *array, int base, int size);
	void _PrintDegrees(int i);
	void _PrintNeighbourhood(int i);
	void _PrintFuzzyNeighbourhood(int i);
	void _PrintSeamnessTable(std::vector<int> *pVector, int iCount);
	void _PrintPartsMatching();
	void saveIntermediateFiles();

	/// Number of parts of two creatures (index the same as for m_Mod).
	int m_aiPartCount[2];

	/// Difference between number of parts in organisms
	int m_iDV;

	/// Sum of absolute values of differences between matched part degrees
	int m_iDD;

	/// Sum of absolute values of differences between matched part 
	/// in neurons number.
	int m_iDN;
	//2 matrices of neighbourhood of parts - one for each genotype 

	/// Sum of Euclidean distances between matched parts
	/// Unmatched Parts have the distance measured to (0,0,0) (the middle of
	/// an organism)
	double m_dDG;

	/// Object that holds the matching of Parts.
	SimilMatching *m_pMatching;
	/// Object that holds temporary matching of Parts.
	SimilMatching *tempMatching;

	/// Type of 4 ints - describing one Part of the creature in
	/// its sorted table of degrees
	/// TDN[0] - original index of creature's Part (that is "i" from GetPart(i))
	/// TDN[1] - degree (number of adjacent joints) of one Part
	/// TDN[2] - number of NeuroConnections and Neurons belonging to one Part
	/// TDN[3] - number of Neurons of the Part
	/// TDN[4] - fuzzy degree
	typedef int TDN[5];

	/** 2 arrays holding information about compared organisms (one for each
	creature) of degree and neuro info for Parts.
	Index corresponds to the one in m_Mod
	m_aDegrees[i][j] is a TDN of the j-th Part of the i-th creature in m_Mod
	*/
	TDN *m_aDegrees[2];

	//std::pair<TDN, double> *m_aDegrees[2];
	/// Holds information on all on-joint neurons. Only TDN[3] and TDN[4]
	/// are important (original index and degree are not important).
	TDN m_aOnJoint[2];

	/// Holds information on all neurons that are not placed neither on
	/// a joint nor on a part. Only TDN[3] and TDN[4]
	/// are important (original index and degree are not important).
	TDN m_aAnywhere[2];

	//array of parts neighbourhood
	int **m_Neighbours[2];
	//array of "fuzzy neigbourhood" for each of organisms. Dimensions: parts_num x fuzzyDepth
	float **m_fuzzyNeighb[2];

	/// Number of weights in the function which evaluates distance.
	static const int iNOFactors; 
};

#endif /* GREEDY_MEASURE_H */

