package com.framsticks.gui.console;

import java.util.LinkedList;
import java.util.List;

import com.framsticks.communication.ClientSideManagedConnection;
import com.framsticks.communication.ClientSideResponseFuture;
import com.framsticks.communication.File;
import com.framsticks.communication.Request;
import com.framsticks.communication.Response;
import com.framsticks.communication.queries.ApplicationRequest;
import com.framsticks.core.Path;
import static com.framsticks.core.TreeOperations.*;
import com.framsticks.gui.SwingDispatcher;
import com.framsticks.params.CompositeParam;
import com.framsticks.params.annotations.AutoAppendAnnotation;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.types.ListParam;
import com.framsticks.remote.RemoteTree;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.Dispatching;
import com.framsticks.util.dispatching.FutureHandler;
import com.framsticks.util.dispatching.RunAt;
import com.framsticks.util.lang.Casting;
import com.framsticks.util.lang.Containers;
import com.framsticks.util.lang.Pair;
import com.framsticks.util.lang.Strings;

@FramsClassAnnotation
public class ManagedConsole extends InteractiveConsole {


	protected RemoteTree tree;

	/**
	 * @param connection
	 */
	public ManagedConsole() {
	}

	/**
	 * @return the connection
	 */
	@Override
	public ClientSideManagedConnection getConnection() {
		return (ClientSideManagedConnection) connection;
	}


	protected void sendImplementation(String line) {
		if (!connection.isConnected()) {
			throw new FramsticksException().msg("not connected").arg("console", this);
		}
		//Move that to managed connection
		ApplicationRequest request;
		try {
			Pair<String, String> command = Strings.splitIntoPair(line, ' ', "");
			request = Casting.throwCast(ApplicationRequest.class, Request.createRequestByTypeString(command.first));
			request.parseRest(command.second);
		} catch (FramsticksException e) {
			throw new FramsticksException().msg("invalid line").arg("line", line).cause(e);
		}

		paintLine(line);

		getConnection().send(request, SwingDispatcher.getInstance(), new ClientSideResponseFuture(this) {
			@Override
			protected void processOk(Response response) {
				for (File f : response.getFiles()) {
					consolePainter.paintMessage(f);
				}
			}
		});
	}


	@Override
	protected void findCompletionPropositions(final String prefix) {
		Pair<CharSequence, CharSequence> command = Request.takeIdentifier(prefix);
		if (command == null) {
			return;
		}

		Casting.throwCast(ApplicationRequest.class, Request.createRequestByTypeString(command.first.toString()));
		// Pair<String, String> rest = Strings
		Pair<CharSequence, CharSequence> rest = Request.takeIdentifier(command.second);
		if (rest == null) {
			List<String> propositions = new LinkedList<String>();
			propositions.add(command.first.toString() + " /");
			processCompletionResult(prefix, propositions);
			return;
		}

		final String textual = rest.first.toString();
		if (!textual.startsWith("/")) {
			throw new FramsticksException().msg("invalid line").arg("line", prefix);
		}
		// final Iterator<String> iterator = Path.splitPath(textual);

		tryGet(tree, textual, new FutureHandler<Path>(this) {

			@Override
			protected void result(final Path path) {
				if (!textual.startsWith(path.getTextual())) {
					throw new FramsticksException().msg("invalid state").arg("line", prefix).arg("path", path);
				}
				assert path.getTree().isActive();

				final Runnable finalizeCompletion = new Runnable() {
					@Override
					public void run() {
						String remaining = textual.substring(path.getTextual().length());

						String base = prefix.substring(0, prefix.length() - (textual.length() - path.getTextual().length()));
						if (path.size() > 1) {
							base = base + "/";
						}

						if (remaining.startsWith("/")) {
							remaining = remaining.substring(1);
						}

						if (remaining.indexOf('/') != -1) {
							/** It is to long. */
							return;
						}
						final List<String> propositions = new LinkedList<String>();
						for (CompositeParam p : Containers.filterInstanceof(bindAccess(path).getParams(), CompositeParam.class)) {
							if (remaining.equals("") || p.getId().startsWith(remaining)) {
								propositions.add(base + p.getId());
							}
						}

						dispatch(new RunAt<ManagedConsole>(ManagedConsole.this) {

							@Override
							protected void runAt() {
								processCompletionResult(prefix, propositions);
							}
						});
					}
				};

				if (path.getTop().getParam() instanceof ListParam) {
					tree.get(path, new FutureHandler<Path>(ManagedConsole.this) {
						@Override
						protected void result(Path result) {
							finalizeCompletion.run();
						}
					});
					return;
				}
				finalizeCompletion.run();

			}
		});


	}

	@AutoAppendAnnotation
	public ManagedConsole setTree(RemoteTree tree) {
		this.tree = tree;
		connection = tree.getConnection();
		return this;
	}

	@Override
	protected void joinableStart() {
		super.joinableStart();
		Dispatching.use(tree, this);
	}

	@Override
	protected void joinableInterrupt() {
		Dispatching.drop(tree, this);
		super.joinableInterrupt();
	}

	@Override
	protected void joinableJoin() throws InterruptedException {
		Dispatching.join(tree);
		super.joinableJoin();
	}
}
