package com.framsticks.util.dispatching;

import java.util.LinkedList;
import java.util.ListIterator;
import java.util.TimerTask;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.params.annotations.AutoAppendAnnotation;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;

@FramsClassAnnotation
public class StackedJoinable extends AbstractJoinable implements JoinableParent {
	private static final Logger log = LogManager.getLogger(StackedJoinable.class);


	protected final LinkedList<Joinable> joinables = new LinkedList<Joinable>();
	protected ListIterator<Joinable> iterator;
	protected Joinable currentJoinable;

	protected double interval = 1;

	/**
	 *
	 */
	public StackedJoinable() {
		super();
	}

	@AutoAppendAnnotation
	public void addJoinable(Joinable joinable) {
		joinables.add(joinable);
	}

	@Override
	public String getName() {
		return "stacked";
	}

	@Override
	public void childChangedState(Joinable joinable, JoinableState state) {
		// log.error("{} notified {} about changing state to {}", joinable, this, state);
		if (state.equals(JoinableState.RUNNING)) {
			if (joinable == currentJoinable) {
				Dispatching.getTimer().schedule(new TimerTask() {
					@Override
					public void run() {
						startNext();
					}
				}, (int)(interval * 1000));
			}
			return;
		}
		if (state.equals(JoinableState.FINISHING)) {
			if (joinable == currentJoinable) {
				Dispatching.getTimer().schedule(new TimerTask() {
					@Override
					public void run() {
						stopNext();
					}
				}, (int)(interval * 1000));
			}
			return;
		}

	}

	protected void startNext() {
		if (!iterator.hasNext()) {
			currentJoinable = null;
			log.debug("started all");
			return;
		}
		currentJoinable = iterator.next();
		log.debug("starting {}", currentJoinable);
		Dispatching.use(currentJoinable, this);
	}

	protected void stopNext() {
		if (!iterator.hasPrevious()) {
			finishJoinable();
			return;
		}
		currentJoinable = iterator.previous();
		log.debug("stoping {}", currentJoinable);
		Dispatching.drop(currentJoinable, this);
	}

	@Override
	protected void joinableStart() {
		iterator = joinables.listIterator();
		startNext();
	}

	@Override
	protected void joinableInterrupt() {
		log.debug("interupting");
		stopNext();
	}

	@Override
	protected void joinableFinish() {
		// here process ends and StackedJoinable transists to the joinable state

	}

	@Override
	protected void joinableJoin() throws InterruptedException {

	}

	/**
	 * @return the interval
	 */
	@ParamAnnotation
	public double getInterval() {
		return interval;
	}

	/**
	 * @param interval the interval to set
	 */
	@ParamAnnotation
	public void setInterval(double interval) {
		this.interval = interval;
	}

	public Joinable get(int number) {
		return joinables.get(number);
	}

	public int size() {
		return joinables.size();
	}
}
