package com.framsticks.gui.console;

import java.awt.AWTKeyStroke;
import java.awt.BorderLayout;
import java.awt.KeyboardFocusManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.font.FontRenderContext;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;

import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.lang.Strings;

@FramsClassAnnotation
public abstract class InteractiveConsole extends Console {

	/**
	 * Text field for typing commands.
	 */
	protected JTextField commandLine;
	protected JButton sendButton;

	protected final LinkedList<String> history = new LinkedList<>();
	protected ListIterator<String> historyCursor;

	public void setCommandLine(String command) {
		commandLine.setText(command);
	}

	/**
	 * @param connection
	 */
	public InteractiveConsole() {
	}

	@Override
	protected void initializeGui() {
		super.initializeGui();

		commandLine = new JTextField();

		commandLine.addKeyListener(new KeyListener() {
			@Override
			public void keyTyped(KeyEvent e) {

			}

			@Override
			public void keyPressed(KeyEvent e) {
				onKeyPressed(e.getKeyCode());
			}

			@Override
			public void keyReleased(KeyEvent e) {
			}
		});

		commandLine.setFocusTraversalKeys(KeyboardFocusManager.FORWARD_TRAVERSAL_KEYS, Collections.<AWTKeyStroke> emptySet());

		completionPopup = new JPopupMenu();

		// completionPopupAction = new AbstractAction() {
		//	@Override
		//	public void actionPerformed(ActionEvent event) {
		//		JMenuItem item = (JMenuItem) event.getSource();
		//		commandLine.setText(item.getText());
		//	}
		// };

		sendButton = new JButton("Send");
		sendButton.setName("send");
		sendButton.addActionListener(new ActionListener() {

			public void actionPerformed(final ActionEvent e) {
				sendCurrent();
			}
		});

		final Box cmdBox = new Box(BoxLayout.LINE_AXIS);
		cmdBox.add(commandLine);
		cmdBox.add(Box.createHorizontalStrut(10));
		cmdBox.add(sendButton);
		cmdBox.setBorder(BorderFactory.createEmptyBorder(0, 7, 7, 7));

		panel.add(cmdBox, BorderLayout.PAGE_END);

		getSwing().addWindowListener(new WindowAdapter() {
			public void windowOpened(WindowEvent e) {
				commandLine.requestFocus();
			}
		});
	}


	protected abstract void findCompletionPropositions(String prefix);

	@SuppressWarnings("serial")
	protected void processCompletionResult(String prefix, List<String> propositions) {
		assert isActive();
		if (propositions.isEmpty()) {
			return;
		}
		if (!commandLine.getText().equals(prefix)) {
			return;
		}
		if (propositions.size() == 1) {
			commandLine.setText(propositions.get(0));
			return;
		}

		Iterator<String> i = propositions.iterator();
		String common = i.next();
		while (i.hasNext()) {
			common = Strings.commonPrefix(common, i.next());
		}
		if (!prefix.equals(common)) {
			commandLine.setText(common);
			return;
		}

		completionPopup.setVisible(false);
		completionPopup.removeAll();

		for (final String c : propositions) {
			completionPopup.add(new JMenuItem(new AbstractAction(c) {
				@Override
				public void actionPerformed(ActionEvent e) {
					commandLine.setText(c);
				}
			}));
		}


		double width = commandLine.getFont().getStringBounds(prefix, new FontRenderContext(null, false, false)).getWidth();
		completionPopup.show(commandLine, (int) width, 0);
		completionPopup.requestFocus();
	}

	protected JPopupMenu completionPopup;


	private void onKeyPressed(int keyCode) {
		if (keyCode == KeyEvent.VK_TAB) {
			try {
				findCompletionPropositions(commandLine.getText());
			} catch (FramsticksException e) {
				handle(e);
			}
			return;
		}

		if (keyCode == KeyEvent.VK_UP || keyCode == KeyEvent.VK_DOWN) {
			boolean up = (keyCode == KeyEvent.VK_UP);
			if (history.isEmpty()) {
				return;
			}

			String line;
			if (up) {
				if (historyCursor == null) {
					historyCursor = history.listIterator(0);
				}
				if (historyCursor.hasNext()) {
					line = historyCursor.next();
				} else {
					historyCursor = null;
					line = "";
				}
			} else {
				if (historyCursor == null) {
					historyCursor = history.listIterator(history.size());
				}
				if (historyCursor.hasPrevious()) {
					line = historyCursor.previous();
				} else {
					historyCursor = null;
					line = "";
				}
			}
			commandLine.setText(line);
			return;
		}

		if (keyCode == KeyEvent.VK_ENTER) {
			sendCurrent();
			return;
		}

	}

	protected abstract void sendImplementation(String line);

	/**
	 * Sends message to manager and adds message to console frame.
	 *
	 * Message is not put into the text area by that method, because
	 * in DirectConsole it is done by means of listening on Connection.
	 */
	public void sendCurrent() {
		assert isActive();
		String line = commandLine.getText();
		history.remove(line);
		history.add(0, line);
		historyCursor = null;
		commandLine.setText("");

		sendImplementation(line);

	}

	/**
	 * Adds query to console window.
	 *
	 * @param line Line of string query.
	 */
	protected void paintLine(String line) {
		consolePainter.userLine(line);
	}


}
