package com.framsticks.parsers;

import org.testng.annotations.*;

import com.framsticks.model.*;
import com.framsticks.model.f0.Schema;
import com.framsticks.model.f0.SchemaBuilder;
import com.framsticks.params.*;
import com.framsticks.params.types.FloatParam;
import com.framsticks.params.types.StringParam;
import com.framsticks.test.TestConfiguration;
import com.framsticks.util.math.Point3d;

import java.io.IOException;
import java.text.ParseException;
import java.util.List;
import static org.fest.assertions.Assertions.*;
import static org.fest.assertions.Delta.*;

/**
 * Author: Piotr Śniegowski
 */
public class F0ParserTest extends TestConfiguration {

	private Schema schema;
	private List<Access> accesses;
	private List<ModelComponent> components;
	private Model model;

	@BeforeClass
	public void setUp() throws Exception {
		schema = new SchemaBuilder().stream(Schema.getDefaultDefinitionAsStream()).finish();
	}

	@Test
	public void checkFramsClasses() {
		FramsClass modelClass = schema.getRegistry().getFramsClass("Model");
		assertThat(modelClass).isNotNull();
		assertThat(modelClass.getParam("se")).isInstanceOf(FloatParam.class);
		// assertThat(modelClass.getParam("energ0")).isInstanceOf(FloatParam.class);
		assertThat(modelClass.getParam("Vstyle")).isInstanceOf(StringParam.class);
		assertThat(modelClass.getParamCount()).isEqualTo(6);
		// assertThat();

	}

	@Test(dependsOnMethods = "checkFramsClasses")
	public void primitiveParam() {
		FramsClass joint = schema.getFramsClass("j");
		PrimitiveParam<?> dx = joint.getParamEntry("dx", PrimitiveParam.class);
		assertThat(dx).isInstanceOf(FloatParam.class);
		assertThat(schema.getNeuroClasses().size()).isEqualTo(21);
		assertThat(dx.getName()).isEqualTo("delta.x");
		assertThat(dx.getMin(Double.class)).isEqualTo(-2.0, delta(0.0));

		assertThat(schema.getRegistry().getFramsClass("n").getParamEntry("d", StringParam.class).getDef(String.class)).isEqualTo("N");
	}

	@Test(dependsOnMethods = "primitiveParam")
	public void readF0() throws IOException, ParseException {
		assertThat(schema.getFramsClass("p")).isInstanceOf(FramsClass.class);
		assertThat(schema.getRegistry().getFramsClass("p").getParamEntry("as", FloatParam.class).getDef(Double.class)).isEqualTo(0.25, delta(0.0));

		accesses = new F0Parser(schema, F0ParserTest.class.getResourceAsStream("/parsers/f0_example.txt")).parse();

		assertThat(accesses.size()).isEqualTo(19);
		assertThat(accesses.get(0).getSelected()).isInstanceOf(Model.class);
		assertThat(accesses.get(5).get("i", String.class)).isEqualTo("1,2,3,\"dsadsa,,,,");
		assertThat(accesses.get(7).get("d", String.class)).isEqualTo("|:p=0.25,r=1");
		assertThat(accesses.get(10).get("d", String.class)).isEqualTo("@:p=0.25");
	}

	@Test(dependsOnMethods = {"readF0"})
	public void stripAccess() {
		components = ParamsUtil.stripAccess(accesses, ModelComponent.class);

		assertThat(components.get(1)).isInstanceOf(Part.class);
		assertThat(components.get(4)).isInstanceOf(Joint.class);
		assertThat(components.get(6)).isInstanceOf(NeuroDefinition.class);
		assertThat(components.get(12)).isInstanceOf(NeuroConnection.class);
	}

	@Test(dependsOnMethods = {"stripAccess"})
	public void buildModel() {
		model = new ModelBuilder().addComponents(components).finish();

		assertThat(model.getParts().size()).isEqualTo(3);
		assertThat(model.getNeuroDefinitions().size()).isEqualTo(6);
		assertThat(model.getJoints().size()).isEqualTo(2);
		assertThat(model.getNeuroConnections().size()).isEqualTo(7);

		assertThat(model.getParts().get(1).getPosition().x).isEqualTo(2.0, delta(0.0));
		assertThat(model.getParts().get(2).getPosition().sub(new Point3d(2.27236, -0.0792596, -0.958924)).length()).isLessThan(0.0001);
		assertThat(model.getParts().get(2).getOrientation().y.sub(new Point3d(0.870277, -0.404792, 0.280644)).length()).isLessThan(0.0001);

		assertThat(model.getJoints().get(0).part1).isEqualTo(0);
		assertThat(model.getJoints().get(0).part2).isEqualTo(1);

		assertThat(model.getNeuroDefinitions().get(0).part).isEqualTo(1);
		assertThat(model.getNeuroDefinitions().get(0).joint).isEqualTo(-1);
		assertThat(model.getNeuroDefinitions().get(1).details).isEqualTo("|:p=0.25,r=1");
		assertThat(model.getNeuroDefinitions().get(3).details).isEqualTo("N");
		assertThat(model.getNeuroDefinitions().get(4).part).isEqualTo(-1);

		assertThat(model.getNeuroConnections().get(2).connectedNeuro).isEqualTo(0);
		assertThat(model.getNeuroConnections().get(5).weight).isEqualTo(5.6, delta(0.0));

	}

	@Test(dependsOnMethods = {"buildModel"})
	public void print() throws Exception {
		ListSink sink = new ListSink();

		new F0Writer(schema, model, sink).write();

		assertThat(sink.getOut()).containsExactly(
			"p:",
			"p:2.0,i=,Vstyle=",
			"p:2.272364001928095,-0.07925961087140347,-0.9589242746631385,i=bla",
			"j:0,1,dx=2.0",
			"j:1,2,rx=8.0,5.0,6.0,dx=1.0,i=\"1,2,3,\\\"dsadsa,,,,\"",
			"n:p=1",
			"n:j=0,d=\"|:p=0.25,r=1\"",
			"n:j=0,d=G",
			"n:p=1",
			"n:j=0,d=@:p=0.25",
			"n:p=1,d=Nu",
			"c:0,2",
			"c:0,2,5.4",
			"c:1,0",
			"c:2,0,3.4",
			"c:0,1,4.5",
			"c:1,0,5.6",
			"c:2,1",
			"m:"
		);

	}
}
