package com.framsticks.params;

import java.util.Arrays;

import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import com.framsticks.params.types.EventParam;
import com.framsticks.params.types.ProcedureParam;
import com.framsticks.params.types.StringParam;
import com.framsticks.parsers.Savers;
import com.framsticks.test.TestChangeEvent;
import com.framsticks.test.TestClass;
import com.framsticks.test.TestConfiguration;
import com.framsticks.util.lang.Holder;

import static org.fest.assertions.Assertions.*;

@Test
public class FramsClassBuilderTest extends TestConfiguration {

	FramsClass framsClass;
	ReflectionAccess access;
	TestClass test;

	@BeforeClass
	public void setUp() {
		test = new TestClass();
		test.setName("test");
		test.appendHistory("first");
		framsClass = FramsClass.build().forClass(TestClass.class);
	}

	@Test
	public void checkProcedureParams() {
		assertThat(framsClass.getParamCount()).isEqualTo(8);

		assertThat(framsClass.getParam("name")).isInstanceOf(StringParam.class);
		assertThat(framsClass.getParam("history")).isInstanceOf(StringParam.class);
		assertThat(framsClass.getParam("history_changed")).isInstanceOf(EventParam.class);

		assertThat(framsClass.getParam("appendHistory")).isInstanceOf(ProcedureParam.class);
		assertThat(framsClass.getParam("resetHistory")).isInstanceOf(ProcedureParam.class);

		ProcedureParam appendHistory = framsClass.getParamEntry("appendHistory", ProcedureParam.class);
		assertThat(appendHistory.getArgumentsType().size()).isEqualTo(1);
		assertThat(appendHistory.getArgumentsType().get(0).getId()).isEqualTo("arg0");
	}

	@Test(dependsOnMethods = "checkProcedureParams")
	public void print() {
		assertThat(Savers.saveFramsClass(new ListSink(), framsClass).getOut()).isEqualTo(
				Arrays.asList(
					"class:",
					"name:TestClass",
					"id:TestClass",
					"",
					"prop:",
					"id:name",
					"name:Name",
					"type:s",
					"",
					"prop:",
					"id:history",
					"name:History",
					"type:s",
					"",
					"prop:",
					"id:history_changed",
					"name:HistoryListener",
					"type:e TestChangeEvent",
					"",
					"prop:",
					"id:appendHistory",
					"name:AppendHistory",
					"type:p d(s arg0)",
					"",
					"prop:",
					"id:resetHistory",
					"name:ResetHistory",
					"type:p()",
					"",
					"prop:",
					"id:children",
					"name:Children",
					"type:l TestChild uid",
					"flags:1",
					"",
					"prop:",
					"id:createChild",
					"name:CreateChild",
					"type:p(s arg0)",
					"",
					"prop:",
					"id:children_changed",
					"name:ChildrenListener",
					"type:e ListChange",
					""
				)
			);
	}

	@Test(dependsOnMethods = "print")
	public void createAccess() {
		access = new ReflectionAccess(TestClass.class, framsClass);
		access.select(test);
	}

	@Test(dependsOnMethods = "createAccess")
	public void callProcedures() {

		assertThat(access.get("history", String.class)).isEqualTo("initial|first|");
		Object result = access.call("appendHistory", new Object[] {"second"});

		assertThat(result).isInstanceOf(Integer.class);
		assertThat(result).isEqualTo(21);
		assertThat(access.get("history", String.class)).isEqualTo("initial|first|second|");

		result = access.call("resetHistory", null);
		assertThat(result).isNull();

		assertThat(access.get("history", String.class)).isEqualTo("");
	}

	@Test(dependsOnMethods = "callProcedures")
	public void listeners() {

		final Holder<String> called = new Holder<>();

		final EventListener<TestChangeEvent> listener = new EventListener<TestChangeEvent>() {

			@Override
			public void action(TestChangeEvent argument) {
				called.set(argument.history);
			}
		};

		final EventParam eventParam = access.getFramsClass().getParamEntry("history_changed", EventParam.class);
		access.reg(eventParam, listener);

		final String currentHistory = access.get("history", String.class);
		final String addition = "test";

		access.call("appendHistory", new Object[] { addition });

		String expected = currentHistory + addition + "|";
		assertThat(access.get("history", String.class)).isEqualTo(expected);
		assertThat(called.get()).isEqualTo(expected);
		access.regRemove(eventParam, listener);
	}

}
