package com.framsticks.running;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;


import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.core.ValueChange;
import com.framsticks.params.EventListener;
import com.framsticks.params.EventListeners;
import com.framsticks.params.ParamFlags;
import com.framsticks.params.annotations.AutoAppendAnnotation;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.Misc;
import com.framsticks.util.dispatching.AbstractJoinable;
import com.framsticks.util.dispatching.Dispatching;
import com.framsticks.util.dispatching.Joinable;
import com.framsticks.util.dispatching.JoinableParent;
import com.framsticks.util.dispatching.JoinableState;
import com.framsticks.util.dispatching.RunAt;
import com.framsticks.util.dispatching.Thread;
import com.framsticks.util.dispatching.ThrowExceptionHandler;
import com.framsticks.util.io.Encoding;

@FramsClassAnnotation
public class ExternalProcess extends AbstractJoinable implements JoinableParent {
	private static final Logger log = LogManager.getLogger(ExternalProcess.class);

	protected List<String> arguments = new ArrayList<>();
	protected Process process;
	protected final ProcessBuilder builder = new ProcessBuilder();
	protected Thread<ExternalProcess> readerThread = new Thread<ExternalProcess>();

	protected PrintWriter input;
	protected BufferedReader output;
	protected Integer exitCode;
	protected String echoInput;

	protected final EventListeners<ValueChange> listeners = new EventListeners<>();

	@AutoAppendAnnotation
	@ParamAnnotation(id = "line_output")
	public void addOutputListener(EventListener<ValueChange> listener) {
		synchronized (listeners) {
			listeners.add(listener);
		}
	}

	@ParamAnnotation(id = "line_output")
	public void removeOutputListener(EventListener<ValueChange> listener) {
		synchronized (listeners) {
			listeners.remove(listener);
		}
	}

	/**
	 *
	 */
	public ExternalProcess() {
		super();
		setName("process");
		arguments.add(null);
		builder.redirectErrorStream(true);
	}

	/**
	 * @return the command
	 */
	@ParamAnnotation(flags = ParamFlags.USERREADONLY)
	public String getCommand() {
		return arguments.get(0);
	}

	/**
	 * @param command the command to set
	 */
	@ParamAnnotation
	public void setCommand(String command) {
		arguments.set(0, command);
	}

	protected void readerTask() {

		log.debug("reading output from " + this);
		String line;
		try {
			try {
				while ((line = output.readLine()) != null) {
					log.trace("read line: {}", line);
					synchronized (listeners) {
						listeners.actionForAll(new ValueChange(line));
					}
				}
			} catch (IOException e) {
				throw new FramsticksException().msg("failed to read line from output of process").cause(e);
			}
			try {
				exitCode = process.waitFor();
			} catch (InterruptedException e) {
				throw new FramsticksException().msg("failed to wait for process").cause(e);
			}
			log.debug("process ended {}", this);
			// process = null;
		} catch (FramsticksException e) {
			log.error("exception caught in process {}", this, e);
		}
		interruptJoinable();
		// finish();
	}

	@ParamAnnotation(flags = ParamFlags.USERREADONLY)
	public void setDirectory(String directory) {
		builder.directory(new File(directory));
	}

	@ParamAnnotation
	public String getDirectory() {
		return builder.directory() != null ? builder.directory().getName() : ".";
	}

	@Override
	protected void joinableStart() {
		log.debug("running process with arguments: {}", arguments);
		builder.command(arguments);
		try {
			process = builder.start();
			input = new PrintWriter(new OutputStreamWriter(process.getOutputStream(), Encoding.getDefaultCharset()));
			output = new BufferedReader(new InputStreamReader(process.getInputStream(), Encoding.getDefaultCharset()));

		} catch (IOException e) {
			throw new FramsticksException().msg("failed to start process").cause(e);
		}

		readerThread.dispatch(new RunAt<ExternalProcess>(ThrowExceptionHandler.getInstance()) {

			@Override
			protected void runAt() {
				readerTask();
			}

		});
		Dispatching.use(readerThread, this);

		if (echoInput != null) {
			input.println(echoInput);
			input.flush();
		}
	}

	@Override
	public String toString() {
		return getName() + "[" + Misc.returnNotNull(getCommand(), "?") + "]";
	}

	/**
	 * @return the input
	 */
	public PrintWriter getInput() {
		return input;
	}

	/**
	 * @return the echoInput
	 */
	@ParamAnnotation(flags = ParamFlags.USERREADONLY)
	public String getEchoInput() {
		return echoInput;
	}

	/**
	 * @param echoInput the echoInput to set
	 */
	@ParamAnnotation
	public void setEchoInput(String echoInput) {
		this.echoInput = echoInput;
	}

	@Override
	protected void joinableInterrupt() {
		process.destroy();
		Dispatching.drop(readerThread, this);
		// finish();
	}

	@Override
	@ParamAnnotation(flags = ParamFlags.USERREADONLY)
	public String getName() {
		return readerThread.getName();
	}

	/**
	 * @param name the name to set
	 */
	@ParamAnnotation
	public void setName(String name) {
		readerThread.setName(name);
	}

	@Override
	protected void joinableFinish() {

	}

	@Override
	protected void joinableJoin() throws InterruptedException {
		Dispatching.join(readerThread);
	}

	@Override
	public void childChangedState(Joinable joinable, JoinableState state) {
		proceedToState(state);
	}



}
