package com.framsticks.params;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.concurrent.Immutable;

import com.framsticks.params.annotations.AutoAppendAnnotation;
import com.framsticks.params.types.EventParam;
import com.framsticks.params.types.ProcedureParam;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.lang.Pair;


import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import static com.framsticks.util.lang.Containers.*;

@Immutable
public class ReflectionAccessBackend {

	private final static Logger log = LogManager.getLogger(ReflectionAccessBackend.class.getName());

	protected static final Map<Pair<Class<?>, FramsClass>, ReflectionAccessBackend> synchronizedCache = Collections.synchronizedMap(new HashMap<Pair<Class<?>, FramsClass>, ReflectionAccessBackend>());


	public interface ReflectedGetter {
		public <T> T get(Object object, Class<T> type) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException;
	}

	public interface ReflectedSetter {
		public <T> void set(Object object, T value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException;
	}

	public interface ReflectedCaller {
		public Object call(Object object, Object[] arguments) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException;
	}

	public interface ReflectedAdder{
		public void reg(Object object, EventListener<?> listener) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException;
	}

	public interface ReflectedRemover{
		public void regRemove(Object object, EventListener<?> listener) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException;
	}

	protected final Map<String, ReflectedSetter> setters = new HashMap<>();
	protected final Map<String, ReflectedGetter> getters = new HashMap<>();
	protected final Map<String, ReflectedCaller> callers = new HashMap<>();
	protected final Map<String, ReflectedAdder> adders = new HashMap<>();
	protected final Map<String, ReflectedRemover> removers = new HashMap<>();

	protected final List<Method> autoAppendMethods = new ArrayList<>();

	/**
	 * @param params
	 */
	public ReflectionAccessBackend() {
	}

	public static ReflectionAccessBackend getOrCreateFor(Class<?> reflectedClass, FramsClass framsClass) {

		Pair<Class<?>, FramsClass> id = new Pair<Class<?>, FramsClass>(reflectedClass, framsClass);
		ReflectionAccessBackend backend = synchronizedCache.get(id);
		if (backend != null) {
			return backend;
		}

		log.debug("constructing backend for {}", id);
		backend = new ReflectionAccessBackend();

		Map<String, ParamCandidate> candidates = ParamCandidate.getAllCandidates(reflectedClass).getCandidates();

		try {
			for (final ProcedureParam pp : filterInstanceof(framsClass.getParamEntries(), ProcedureParam.class)) {
				if (!candidates.containsKey(pp.getId())) {
					log.trace("java class does implement method {}", pp);
					continue;
				}
				ParamCandidate pc = candidates.get(pp.getId());
				final Method method = pc.getCaller();

				backend.callers.put(pp.getId(), new ReflectedCaller() {

					@Override
					public Object call(Object object, Object[] arguments) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
						return method.invoke(object, arguments);
					}
				});

			}

			for (final EventParam ep : filterInstanceof(framsClass.getParamEntries(), EventParam.class)) {
				if (!candidates.containsKey(ep.getId())) {
					log.trace("java class does not implement the event param {}", ep);
					continue;
				}
				ParamCandidate ec = candidates.get(ep.getId());
				final Method adder = ec.getAdder();
				final Method remover = ec.getRemover();

				backend.adders.put(ep.getId(), new ReflectedAdder() {

					@Override
					public void reg(Object object, EventListener<?> listener) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
						adder.invoke(object, listener);
					}
				});

				backend.removers.put(ep.getId(), new ReflectedRemover() {

					@Override
					public void regRemove(Object object, EventListener<?> listener) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
						remover.invoke(object, listener);
					}
				});
			}

			for (final ValueParam vp : filterInstanceof(framsClass.getParamEntries(), ValueParam.class)) {
				if (!candidates.containsKey(vp.getId())) {
					throw new ConstructionException().msg("missing candidate for param").arg("param", vp);
				}
				ParamCandidate pc = candidates.get(vp.getId());
				if (pc.isReadOnly() && !vp.hasFlag(ParamFlags.READONLY)) {
					throw new ConstructionException().msg("readonly state conflict").arg("param", vp);
				}
				if (!typeMatch(pc.getRawType(), vp.getStorageType())) {
					throw new ConstructionException().msg("types mismatch for param").arg("param", vp).arg("candidate", pc.getType()).arg("storage", vp.getStorageType());
				}

				final boolean primitive = pc.isPrimitive();
				if (pc.getField() != null) {
					final Field f = pc.getField();
					backend.getters.put(vp.getId(), new ReflectedGetter() {
						@Override
						public <T> T get(Object object, Class<T> type) throws IllegalArgumentException, IllegalAccessException {
							return type.cast(f.get(object));
						}
					});
					if (!pc.isFinal()) {
						backend.setters.put(vp.getId(), new ReflectedSetter() {
							@Override
							public <T> void set(Object object, T value) throws IllegalArgumentException, IllegalAccessException {
								if (value == null && primitive) {
									throw new FramsticksException().msg("setting null to primitive value");
								}
								f.set(object, value);
							}
						});
					}
				} else {
					final Method g = pc.getGetter();

					backend.getters.put(vp.getId(), new ReflectedGetter() {
						@Override
						public <T> T get(Object object, Class<T> type) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
							return type.cast(g.invoke(object));
						}
					});

					if (!pc.isFinal()) {
						final Method s = pc.getSetter();
						backend.setters.put(vp.getId(), new ReflectedSetter() {
							@Override
							public <T> void set(Object object, T value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
								if (value == null && primitive) {
									throw new FramsticksException().msg("setting null to primitive value");
								}
								s.invoke(object, value);
							}
						});
					}
				}
			}
		} catch (ConstructionException e) {
			throw e.arg("java class", reflectedClass).arg("framsClass", framsClass);
		}

		Class<?> javaClass = reflectedClass;
		while (javaClass != null) {

			for (Method m : javaClass.getDeclaredMethods()) {
				AutoAppendAnnotation a = m.getAnnotation(AutoAppendAnnotation.class);
				if (a == null) {
					continue;
				}
				Class<?>[] args = m.getParameterTypes();
				if (args.length != 1) {
					throw new ConstructionException().msg("invalid number of arguments in AutoAppend marked method").arg("method", m).arg("arguments", args.length);
				}
				backend.autoAppendMethods.add(m);
			}

			javaClass = javaClass.getSuperclass();
		}

		Collections.sort(backend.autoAppendMethods, new Comparator<Method>() {

			@Override
			public int compare(Method m0, Method m1) {
				Class<?> arg0 = m0.getParameterTypes()[0];
				Class<?> arg1 = m1.getParameterTypes()[0];
				if (arg0.isAssignableFrom(arg1)) {
					return 1;
				}
				if (arg1.isAssignableFrom(arg0)) {
					return -1;
				}
				return 0;
			}
		});

		synchronizedCache.put(id, backend);
		return backend;
	}

	public static boolean typeMatch(Class<?> a, Class<?> b) {
		if (b.isPrimitive()) {
			throw new FramsticksException().msg("failed to match type, right argument is primitive").arg("left", a).arg("right", b);
		}
		if (!a.isPrimitive()) {
			return a.equals(b);
		}

		if (a.equals(int.class)) {
			return b.equals(Integer.class);
		}
		if (a.equals(double.class)) {
			return b.equals(Double.class);
		}
		if (a.equals(boolean.class)) {
			return b.equals(Boolean.class);
		}
		throw new FramsticksException().msg("failed to match types").arg("left", a).arg("right", b);
	}

}
