package com.framsticks.params;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.util.FramsticksException;
import com.framsticks.util.lang.Containers;
import com.framsticks.util.lang.Holder;
// import com.framsticks.util.lang.Containers;

import static com.framsticks.params.SetStateFlags.*;

public final class AccessOperations {

	private final static Logger log = LogManager.getLogger(SimpleAbstractAccess.class.getName());

	/**
	 *
	 */
	private AccessOperations() {
	}

	/**
	 * Simple String key, value class.
	 */
	public static class Entry {

		public final String key;
		public final String value;

		public Entry(String key, String value) {
			this.key = key;
			this.value = value;
		}

		@Override
		public String toString() {
			return key + " = " + value;
		}
	}

	private static Entry readEntry(Source source) {

		String line;
		String key = null;
		StringBuilder value = null;
		while ((line = source.readLine()) != null) {
			if (key == null) {
				int colonIndex = line.indexOf(':');
				if (colonIndex == -1) {
					return null;
				}
				key = line.substring(0, colonIndex);
				String inlineValue = line.substring(colonIndex + 1);


				if (!inlineValue.startsWith("~")) {
					return new Entry(key, inlineValue);
				}
				value = new StringBuilder();
				value.append(inlineValue.substring(1));
				continue;
			}
			if (value.length() != 0) {
				value.append(System.getProperty("line.separator"));
			}
			if (line.endsWith("~") && !line.endsWith("\\~")) {
				value.append(line.substring(0, line.length() - 1));
				return new Entry(key, value.toString().replaceAll("\\\\~", "~"));
			}
			value.append(line);
		}
		return null;
	}

	public static void load(Access access, Source source) {
		Entry entry;
		while ((entry = readEntry(source)) != null) {
			Param param = access.getParam(entry.key);
			if (param == null) {
				throw new FramsticksException().msg("param not found in access").arg("name", entry.key).arg("access", access);
			}
			if (!(param instanceof ValueParam)) {
				throw new FramsticksException().msg("param is not a value param").arg("param", param).arg("access", access);
			}
			if ((param.getFlags() & ParamFlags.DONTLOAD) == 0) {
				int retFlags = access.set((ValueParam) param, entry.value);
				if ((retFlags & (PSET_HITMIN | PSET_HITMAX)) != 0) {
					String which = ((retFlags & PSET_HITMIN) != 0) ? "small" : "big";
					log.warn("value of key '{}' was too {}, adjusted", entry.key, which);
				}
			}
		}

	}

	public interface Adjuster {
		public Holder<Object> adjust(ValueParam param);
		public Class<? extends ValueParam> getParamType();
	}

	public static class MinAdjuster implements Adjuster {

		/**
		 *
		 */
		public MinAdjuster() {
		}

		@Override
		public Class<? extends ValueParam> getParamType() {
			return PrimitiveParam.class;
		}

		@Override
		public Holder<Object> adjust(ValueParam param) {
			Object value = ((PrimitiveParam<?>) param).getMin(Object.class);
			if (value == null) {
				return null;
			}
			return Holder.make(value);
		}
	}

	public static class MaxAdjuster implements Adjuster {

		/**
		 *
		 */
		public MaxAdjuster() {
		}

		@Override
		public Class<? extends ValueParam> getParamType() {
			return PrimitiveParam.class;
		}

		@Override
		public Holder<Object> adjust(ValueParam param) {
			Object value = ((PrimitiveParam<?>) param).getMax(Object.class);
			if (value == null) {
				return null;
			}
			return Holder.make(value);
		}
	}

	public static class DefAdjuster implements Adjuster {

		protected final boolean numericOnly;

		/**
		 * @param numericOnly
		 */
		public DefAdjuster(boolean numericOnly) {
			this.numericOnly = numericOnly;
		}

		public Class<? extends ValueParam> getParamType() {
			return ValueParam.class;
		}

		@Override
		public Holder<Object> adjust(ValueParam param) {
			if (numericOnly && !(param.isNumeric())) {
				return null;
			}
			return Holder.make(param.getDef(Object.class));
		}
	}

	public static void adjustAll(Access access, Adjuster adjuster) {
		for (ValueParam param : Containers.filterInstanceof(access.getParams(), adjuster.getParamType())) {
			Holder<Object> value = adjuster.adjust(param);
			if (value != null) {
				access.set(param, value.get());
			}
		}
	}

}
