package com.framsticks.communication;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.framsticks.communication.queries.*;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.lang.Pair;
import com.framsticks.util.lang.Strings;

/**
 * Class stores information about query sent to manager.
 */
public abstract class Request {

	public abstract String getCommand();

	protected abstract StringBuilder construct(StringBuilder buffer);




	public static void quoteValue(StringBuilder builder, String value) {
		String quote = ((value.indexOf(' ') > 0) || (value.length() == 0) ? "\"" : "");
		builder.append(quote).append(value).append(quote);
	}

	public static Request parse(CharSequence type, CharSequence rest) {
		final Request request = Request.createRequestByTypeString(type.toString());
		request.parseRest(rest);
		return request;
	}

	public static Request createRequestByTypeString(String type) {
		switch (type) {
			case "get": return new GetRequest();
			case "set": return new SetRequest();
			case "info": return new InfoRequest();
			case "call": return new CallRequest();
			case "reg": return new RegisterRequest();
			case "use": return new UseRequest();
			case "version": return new VersionRequest();
		}
		throw new FramsticksException().msg("unknown request type").arg("type", type);
	}

	public abstract CharSequence parseRest(CharSequence rest);

	@Override
	public String toString() {
		// return getCommand();
		return "request " + stringRepresentation();
	}

	public String stringRepresentation() {
		return construct(new StringBuilder().append(getCommand()).append(" ")).toString();
	}

	public static final Pattern EVENT_PATTERN = Pattern.compile("^\\s*(\\S+)\\s*(\\S+)");

	public static final Pattern STRING_BREAKER_PATTERN = Pattern.compile("^\\s*(?:(?:\\\"([^\"]*)\\\")|([^ ]+))\\s*(.*)$");
	public static final Pattern IDENTIFIER_BREAKER_PATTERN = Pattern.compile("^\\s*([^ ]+)\\s*(.*)$");

	// public static Matcher match(Pattern pattern, CharSequence line) {
	// 	Matcher matcher = pattern.matcher(line);
	// 	if (!matcher.matches()) {
	// 		return null;
	// 		// throw new FramsticksException().msg("failed to take").arg("pattern", pattern).arg("line", line);
	// 	}
	// 	return matcher;
	// }

	public static Pair<CharSequence, CharSequence> takeIdentifier(CharSequence line) {
		Matcher matcher = IDENTIFIER_BREAKER_PATTERN.matcher(line);
		if (!matcher.matches()) {
			return null;
		}
		return new Pair<CharSequence, CharSequence>(line.subSequence(matcher.start(1), matcher.end(1)), line.subSequence(matcher.start(2), matcher.end(2)));
	}


	public static Pair<CharSequence, CharSequence> takeString(CharSequence line) {
		Matcher matcher = STRING_BREAKER_PATTERN.matcher(line);
		if (!matcher.matches()) {
			return null;
		}
		assert ((matcher.start(1) == -1) != (matcher.start(2) == -1));
		return new Pair<CharSequence, CharSequence>(Strings.takeGroup(line, matcher, (matcher.start(1) != -1 ? 1 : 2)), Strings.takeGroup(line, matcher, 3));
	}

	protected static final Pattern REQUEST_ID_BREAKER_PATTERN = Pattern.compile("^\\s*(-?[0-9]+)\\s*(.*)$");

	protected final static Pair<Integer, CharSequence> takeRequestId(boolean withId, CharSequence line) {
		if (withId) {
			Matcher matcher = REQUEST_ID_BREAKER_PATTERN.matcher(line);
			if (!matcher.matches()) {
				return null;
			}
			return new Pair<Integer, CharSequence>(Integer.valueOf(Strings.takeGroup(line, matcher, 1).toString()), Strings.takeGroup(line, matcher, 2));
		}
		return new Pair<Integer, CharSequence>(null, line);
	}

	public static StringBuilder quoteArgumentIfNeeded(StringBuilder builder, Object argument) {
		String r = argument.toString();
		if (r.indexOf(' ') != -1) {
			builder.append('"').append(r).append('"');
		} else {
			builder.append(r);
		}
		return builder;
	}

}
