/*
 * (C) Poznan University of Technology
 * File name: Main.java.
 * Created on 2003-11-08 11:59:53.
 */
package frams_client_3d;

import java.awt.Color;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Vector;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

import foxtrot.Task;
import foxtrot.Worker;

/**
 * The client main window class.
 * @author <a href="mailto:momat@man.poznan.pl">MoMaT</a>
 */
public class MainWindow extends JFrame {

	private Communication comm = new Communication();
	private Log networkLogs = null;
	private boolean listening = false;
	
	private boolean showSelected = true; 
	
	private ArrayList creatureEvents = new ArrayList();
	private String messageEvent = null;
	private String runningEvent = null;
	private String groupEvent = null;
	
	private Vector creaturesCount = new Vector();

	private ArrayList groups = new ArrayList();
	private ArrayList creatures = new ArrayList();
	private ArrayList messages = new ArrayList();
	private boolean running;

	private javax.swing.JPanel jContentPane = null;
	private javax.swing.JSplitPane jSplitPane = null;
	private javax.swing.JSplitPane jSplitPane1 = null;
	private frams_client_3d.Options options = null;
	private frams_client_3d.ViewGL viewGL = null;
	private frams_client_3d.StatusBar statusBar = null;

	public static void main(String[] args) {
		final MainWindow frame = new MainWindow();

		while (true) {
			if (frame.isListening()) {
				HashMap data = null;
				try {
					//handle simulation messages
					if (frame.getMessages()) {
						Runnable runUpdateMessages = new Runnable() {
							public void run() {
								frame.updateMessages();
							}
						};
						SwingUtilities.invokeLater(runUpdateMessages);
					}
					//frame.comm.getLog().clear();

					//handle population changes					
					if (frame.getChangedGroups() || frame.getChangedCreatures()) {
						Runnable runUpdateTree = new Runnable() {
							public void run() {
								frame.updateTree();
							}
						};
						SwingUtilities.invokeLater(runUpdateTree);
					} else {
						//frame.comm.getLog().clear();
					}					
					
					//handle running state changes
					if (frame.getChangedRunningState()) {
						Runnable runUpdateRunningState = new Runnable() {
							public void run() {
								frame.updateRunningState();
							}
						};
						SwingUtilities.invokeLater(runUpdateRunningState);
					} else {
						//frame.comm.getLog().clear();
					}										
				} catch (IOException e) {
					frame.handleIOException(e);
				} catch (CommunicationErrorException e) {
					frame.handleCommunicationErrorException(e);
				}
			} else {
				try {
					Thread.sleep(500);
				} catch (InterruptedException e) {
				}
			}
		}
	}
	/**
	 * This is the default constructor
	 */
	public MainWindow() {
		super();
		initialize();
		setDefaultCloseOperation(EXIT_ON_CLOSE);
		statusBar.addMessage("Application started.", StatusBar.DEFAULT);
	}
	/**
	 * This method initializes this
	 * 
	 * @return void
	 */
	private void initialize() {
		this.setSize(640, 480);
		this.setContentPane(getJContentPane());
		this.setTitle("Frams Client 3D");
		this.setVisible(true);

		networkLogs = new Log();
	}
	/**
	 * This method initializes jContentPane
	 * 
	 * @return javax.swing.JPanel
	 */
	private javax.swing.JPanel getJContentPane() {
		if (jContentPane == null) {
			jContentPane = new javax.swing.JPanel();
			jContentPane.setLayout(new java.awt.BorderLayout());
			jContentPane.add(getJSplitPane(), java.awt.BorderLayout.CENTER);
		}
		return jContentPane;
	}
	/**
	 * This method initializes jSplitPane
	 * 
	 * @return javax.swing.JSplitPane
	 */
	private javax.swing.JSplitPane getJSplitPane() {
		if (jSplitPane == null) {
			jSplitPane = new javax.swing.JSplitPane();
			jSplitPane.setOrientation(javax.swing.JSplitPane.VERTICAL_SPLIT);
			jSplitPane.setTopComponent(getJSplitPane1());
			jSplitPane.setBottomComponent(getStatusBar());
			jSplitPane.setOneTouchExpandable(true);
			jSplitPane.setContinuousLayout(true);
			jSplitPane.setDividerSize(8);
			jSplitPane.setResizeWeight(1.0);
		}
		return jSplitPane;
	}
	/**
	 * This method initializes jSplitPane1
	 * 
	 * @return javax.swing.JSplitPane
	 */
	private javax.swing.JSplitPane getJSplitPane1() {
		if (jSplitPane1 == null) {
			jSplitPane1 = new javax.swing.JSplitPane();
			jSplitPane1.setOrientation(javax.swing.JSplitPane.HORIZONTAL_SPLIT);
			jSplitPane1.setLeftComponent(getOptions());
			jSplitPane1.setRightComponent(getViewGL());
			jSplitPane1.setOneTouchExpandable(true);
			jSplitPane1.setContinuousLayout(true);
			jSplitPane1.setDividerSize(8);
			jSplitPane1.setResizeWeight(0.0);
			jSplitPane1.setBorder(null);
		}
		return jSplitPane1;
	}
	/**
	 * This method initializes options
	 * 
	 * @return frams_client_3d.Options
	 */
	public frams_client_3d.Options getOptions() {
		if (options == null) {
			options = new frams_client_3d.Options();
			options.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
				public void propertyChange(java.beans.PropertyChangeEvent e) {
					if ((e.getPropertyName().equals("connected"))) {
						connect(e.getNewValue());
					} else if ((e.getPropertyName().equals("started"))) {
						startSimulation();
					} else if ((e.getPropertyName().equals("stopped"))) {
						stopSimulation();
					} else if ((e.getPropertyName().equals("logs"))) {
						networkLogs.setVisible(true);
					} else if ((e.getPropertyName().equals("tree"))) {
						passObjectsToView();
					} else if ((e.getPropertyName().equals("show"))) {
						showSelected = e.getNewValue().toString().equals("true");
						passObjectsToView();
					}
				}
			});
		}
		return options;
	}
	/**
	 * This method initializes statusBar
	 * 
	 * @return frams_client_3d.StatusBar
	 */
	private frams_client_3d.StatusBar getStatusBar() {
		if (statusBar == null) {
			statusBar = new frams_client_3d.StatusBar();
		}
		return statusBar;
	}
	/**
	 * This method initializes viewGL
	 * 
	 * @return frams_client_3d.ViewGL
	 */
	private frams_client_3d.ViewGL getViewGL() {
		if (viewGL == null) {
			viewGL = new frams_client_3d.ViewGL();
			viewGL.init();
		}
		return viewGL;
	}
	/**
	 * Connect/disconnect action.
	 * @param value true for connection, false for disconnection
	 */
	private void connect(Object value) {
		if (value.toString().equals("true")) {
			try {
				Worker.post(new Task() {
					public Object run() throws IOException {
						comm.connect(options.getIp(), options.getPort());
						return null;
					}
				});
			} catch (Exception e) {
				statusBar.addMessage("Couldn't get I/O for the connection to a server at "
					+ options.getIp(), StatusBar.ERROR);
				options.setConnected(false);
				return;
			}
			statusBar.addMessage("Connected.", StatusBar.DEFAULT);
			networkLogs.addLogLine("[" + options.getIp() + "]", Color.black);
			readInitialData();
		} else {
			try {
				comm.disconnect();
				options.setConnected(false);
				statusBar.addMessage("Disconnected.", StatusBar.DEFAULT);
				networkLogs.addLogLine("---", Color.blue);

				//stop events listening and clear events lists 
				setListening(false);
				creatureEvents.clear();
				messageEvent = null;
				runningEvent = null;
			} catch (IOException e) {
				statusBar.addMessage("Stream closing error: " + e.getMessage(),
					StatusBar.ERROR);
				//@TODO review: do something more?
				return;
			}
		}
		updateLog();
	}
	/**
	 * Read terrain map and creatures lists data.
	 */
	private void readInitialData() {

		statusBar.addMessage("Reading heighfield...", StatusBar.INFO);
		ArrayList[] heighfield = null;
		try {
			heighfield = (ArrayList[]) Worker.post(new Task() {
				public Object run()
					throws IOException, CommunicationErrorException {
					registerOnMessages();
					registerOnRunningChanged();
					return readHeighfield();
				}
			});
		} catch (IOException e) {
			handleIOException(e);
			return;
		} catch (CommunicationErrorException e) {
			handleCommunicationErrorException(e);
			return;
		} catch (Exception e) {
			handleException(e);
			return;
		}
		viewGL.setHeighfield(heighfield[0], heighfield[1]);
		statusBar.addMessage("Heighfield set.", StatusBar.DEFAULT);
		updateLog();

		statusBar.addMessage("Reading creatures tree...", StatusBar.INFO);
		DefaultTreeModel treeModel = null;
		try {
			treeModel = (DefaultTreeModel) Worker.post(new Task() {
				public Object run()
					throws IOException, CommunicationErrorException {
					return buildTree();
				}
			});
		} catch (IOException e) {
			handleIOException(e);
			return;
		} catch (CommunicationErrorException e) {
			handleCommunicationErrorException(e);
			return;
		} catch (Exception e) {
			handleException(e);
			return;
		}
		options.setTree(treeModel);
		statusBar.addMessage("Creatures tree set.", StatusBar.DEFAULT);
		updateLog();

		statusBar.addMessage("Reading simulation state...", StatusBar.INFO);
		Integer isRunning = null;
		try {
			isRunning = (Integer) Worker.post(new Task() {
				public Object run()
					throws IOException, CommunicationErrorException {
					registerOnGroupsChanged();
					return readRunningState();
				}
			});
		}  catch (IOException e) {
			handleIOException(e);
			return;
		} catch (CommunicationErrorException e) {
			handleCommunicationErrorException(e);
			return;
		} catch (Exception e) {
			handleException(e);
			return;
		}		
		running = isRunning.intValue() == 1;
		updateRunningState();				
		
		setListening(true);
	}
	/**
	 * Communication error handling.
	 * @param e catched exception
	 */		
	private void handleIOException(IOException e) {		
		statusBar.addMessage("I/O error: " + e.getMessage(), StatusBar.ERROR);
		updateLog();
		connect("false");
	}
	/**
	 * Protocol error handling.
	 * @param e catched exception
	 */
	private void handleCommunicationErrorException(CommunicationErrorException e) {
		statusBar.addMessage("protocol error: " + e.getMessage(), StatusBar.WARNING);
		updateLog();
	}
	/**
	 * Error handling for unexpected exceptions.
	 * @param e catched exception
	 */
	private void handleException(Exception e) {
		statusBar.addMessage("unexpected exception: " + e.getMessage(),	StatusBar.ERROR);
		updateLog();
		connect("false");	
	}			
	/**
	 * Simulation begin action.
	 */
	private void startSimulation() {
		String request = "set /simulator running 1";
		try {
			sendRequest(request);
		} catch (IOException e) {
			handleIOException(e);
			return;
		} catch (CommunicationErrorException e) {
			handleCommunicationErrorException(e);
			return;
		}
		updateLog();
	}
	/**
	 * Simulation end action.
	 */
	private void stopSimulation() {
		String request = "set /simulator running 0";
		try {
			sendRequest(request);
		} catch (IOException e) {
			handleIOException(e);
			return;
		} catch (CommunicationErrorException e) {
			handleCommunicationErrorException(e);
			return;
		}
		updateLog();
	}
	/**
	 * Read if simulation is running.
	 * @return Integer initialized to the value of a running property
	 */
	private Integer readRunningState()
	throws IOException, CommunicationErrorException {
		String line = null;
		String request = null;
		ArrayList response = null;		
		
		String state = null;		
		request = "get /simulator running";
		response = sendRequest(request);
		for (int i = 0; i < response.size(); ++i) {
			line = response.get(i).toString();
			if (line.startsWith("running")) {
				state = line.substring(line.indexOf(':') + 1);
				break;
			}
		}
		return Integer.valueOf(state);
	}	
	/**
	 * Read terrain map.
	 * @throws CommunicationErrorException
	 */
	private ArrayList[] readHeighfield()
	throws IOException, CommunicationErrorException {
		String line = null;
		String request = null;
		ArrayList response = null;

		int state = 0;
		ArrayList heighfield[] = new ArrayList[2];
		ArrayList points = new ArrayList();
		ArrayList faces = new ArrayList();

		request = "get /simulator/world faces";
		response = sendRequest(request);
		for (int i = 0; i < response.size(); ++i) {
			line = response.get(i).toString();
			if (line.startsWith("p")) {
				state = 1;
				continue;
			}
			if (line.startsWith("f") && 1 == state) {
				state = 2;
				continue;
			}
			if (line.startsWith("~")) {
				break;
			}
			if (1 == state) {
				String[] parse = line.split(" ");
				points.add(parse);
			}
			if (2 == state) {
				String[] parse = line.split(" ");
				faces.add(parse);
			}
		}
		heighfield[0] = points;
		heighfield[1] = faces;
		return heighfield;
	}
	/**
	 * Read creatures and build the tree.
	 * @return tree model
	 */
	private DefaultTreeModel buildTree()
	throws IOException, CommunicationErrorException {
		String line = null;
		String request = null;
		ArrayList response = null;
		String name = null;

		creaturesCount.clear();
		DefaultMutableTreeNode rootNode = new DefaultMutableTreeNode("world");

		//add non-empty groups to root
		request = "get /simulator/populations/groups name,creatures";
		response = sendRequest(request);
		for (int i = 0; i < response.size(); ++i) {
			line = response.get(i).toString();
			if (line.startsWith("name")) {
				name = line.substring(line.indexOf(':')+1);
			} else if (line.startsWith("creatures")) {
				Integer number = Integer.valueOf(line.substring(line.indexOf(':')+1));
				creaturesCount.add(number);
				if (0 < number.intValue()) {					
					rootNode.add(new DefaultMutableTreeNode(name));
				} else {
					DefaultMutableTreeNode node = new DefaultMutableTreeNode(name);
					rootNode.add(node);
					node.add(new DefaultMutableTreeNode("[empty]"));					
				}
			}
		}

		//add creatures to groups
		for (int i = 0; i < rootNode.getChildCount(); ++i) {
			DefaultMutableTreeNode node = (DefaultMutableTreeNode)rootNode.getChildAt(i);
			int count = ((Integer)creaturesCount.get(i)).intValue();

			//get creatures data			
			for (int j = 0; j < count; ++j) {
				Creature creature = readCreature(i, j);
				node.add(new DefaultMutableTreeNode(creature));
			}
			
			registerOnCreaturesChanged(i, true);
		}
		return new DefaultTreeModel(rootNode);
	}
	/**
	 * Read the creature data from a server.
	 * @param group creature group
	 * @param index creature index in the group
	 * @return creature object
	 * @throws IOException
	 * @throws CommunicationErrorException
	 */
	private Creature readCreature(int group, int index)
	throws IOException, CommunicationErrorException {
		int partsCount = 0;
		int jointsCount = 0;
		String name = null;
		String genotype = null;
		String line = null;
		String request = null;
		ArrayList response = null;

		String requestPrefix = "get /simulator/populations/groups/" + group
			+ "/creatures/"	+ index;

		//get basics
		request = requestPrefix + " name,genotype,parts,joints";
		response = sendRequest(request);
		for (int k = 0; k < response.size(); ++k) {
			line = response.get(k).toString();
			if (line.startsWith("name")) {
				name = line.substring(line.indexOf(':') + 1);
			} else if (line.startsWith("genotype")) {
				genotype = line.substring(line.indexOf(':') + 1);
			} else if (line.startsWith("parts")) {
				partsCount = Integer.parseInt(line.substring(line.indexOf(':') + 1));
			} else if (line.startsWith("joints")) {
				jointsCount = Integer.parseInt(line.substring(line.indexOf(':') + 1));
			}
		}
		Creature creature =	new Creature(name, genotype, partsCount, jointsCount,
			group, index);

		//get joints
		int p1 = 0;
		int p2 = 0;
		request = requestPrefix + "/joints p1,p2";
		response = sendRequest(request);
		for (int k = 0; k < response.size(); ++k) {
			line = response.get(k).toString();
			if (line.startsWith("p1")) {
				p1 = Integer.parseInt(line.substring(line.indexOf(':')+1));
			} else if (line.startsWith("p2")) {
				p2 = Integer.parseInt(line.substring(line.indexOf(':')+1));
				creature.addJoint(p1, p2);
			}
		}

		//get mechparts
		float x = 0;
		float y = 0;
		float z = 0;
		request = requestPrefix + "/mechparts x,y,z";
		response = sendRequest(request);
		for (int k = 0; k < response.size(); ++k) {
			line = response.get(k).toString();
			if (line.startsWith("x")) {
				x = Float.parseFloat(line.substring(line.indexOf(':')+1));
			} else if (line.startsWith("y")) {
				y = Float.parseFloat(line.substring(line.indexOf(':')+1));
			} else if (line.startsWith("z")) {
				z = Float.parseFloat(line.substring(line.indexOf(':')+1));
				creature.addPart(x, y, z);
			}
		}
		return creature;
	}
	/**
	 * Read a group name from a server.
	 * @param index group index in populations
	 * @return a name of the group
	 * @throws IOException
	 * @throws CommunicationErrorException
	 */
	private String readGroup(int index)
	throws IOException, CommunicationErrorException {
		String name = null;
		String line = null;
		String request = null;
		ArrayList response = null;

		request = "get /simulator/populations/groups/" + index + " name";
		response = sendRequest(request);
		for (int k = 0; k < response.size(); ++k) {
			line = response.get(k).toString();
			if (line.startsWith("name")) {
				name = line.substring(line.indexOf(':') + 1);
				break;
			}
		}
		return name;
	}	
	/**
	 * Register for simulator messages.
	 */
	private void registerOnMessages()
	throws IOException, CommunicationErrorException {
		String request = "reg /cli/messages 1";
		ArrayList response = sendRequest(request);
		String parse[] = response.get(0).toString().split("  ");
		messageEvent = parse[1];
	}
	/**
	 * Register for creatures list change in given group.
	 * @param group index of a group
	 */	
	private void registerOnCreaturesChanged(int group, boolean isAdd)
	throws IOException, CommunicationErrorException {
		String request = "reg /simulator/populations/groups/" + group + "/creatures_changed 1";
		ArrayList response = sendRequest(request);
		String parse[] = response.get(0).toString().split("  ");
		if (isAdd) {
			creatureEvents.add(group, parse[1]);
		} else {
			creatureEvents.set(group, parse[1]);
		}
	}
	/**
	 * Register for simulation start/stop change.
	 */
	private void registerOnRunningChanged()
	throws IOException, CommunicationErrorException {
		String request = "reg /simulator/running_changed 1";
		ArrayList response = sendRequest(request);
		String parse[] = response.get(0).toString().split("  ");
		runningEvent = parse[1];
	}
	/**
	 * Register for groups change.
	 */
	private void registerOnGroupsChanged()
	throws IOException, CommunicationErrorException {		
		String request = "reg /simulator/populations/groups_changed 1";
		ArrayList response = sendRequest(request);
		String parse[] = response.get(0).toString().split("  ");
		groupEvent = parse[1];
	}	
	/**
	 * Send request to server and receive a respond.
	 * @param request
	 * @return response
	 */
	private ArrayList sendRequest(String request)
	throws IOException, CommunicationErrorException {
		comm.sendMessage(request);
		ArrayList respond = null;
		try {
			respond = comm.readMessage();
		} catch (InterruptedException e) {
			return new ArrayList();
		}
		return respond;
	}
	/**
	 * Update network communication log.
	 */
	public void updateLog() {
		ArrayList log = comm.getLog();
		for (int i = 0; i < log.size(); ++i) {
			String line = log.get(i).toString();
			if (line.startsWith("> ")) {
				networkLogs.addLogLine(line, Color.gray);
			} else {
				networkLogs.addLogLine(line, Color.darkGray);
			}
		}
		log.clear();
	}
	/**
	 * Pass objects to the viewGL window (according to GUI option all creatures
	 * are passed or just those selected on a creatures tree. 
	 */
	public void passObjectsToView() {
		if (showSelected) {
			viewGL.setCreatures(options.getSelection());
		} else {
			viewGL.setCreatures(options.getAllCreatures());
		}
	}
	/**
	 * Listening property getter.
	 * @return true if events listening has been started
	 */
	synchronized public boolean isListening() {
		return listening;
	}
	/**
	 * Listening property setter.
	 * @param state new value
	 */
	synchronized public void setListening(boolean state) {
		listening = state;
	}
	/**
	 * Read changes on event-registered creatures lists.
	 * @return true if any changes read
	 * @throws IOException Thrown on I/O error
	 * @throws CommunicationErrorException Thrown on protocol error
	 */
	public boolean getChangedCreatures()
		throws IOException, CommunicationErrorException {
		String line = null;
		String request = null;
		ArrayList response = null;
		
		//check registered events
		for (int i = 0; i < creatureEvents.size(); ++i) {			
			Integer type = null;
			Integer group = null;
			
			request = "call " + creatureEvents.get(i) + " sendall";
			response = sendRequest(request);

			//get positions of changes for each group
			for (int k = 0; k < response.size(); ++k) {
				line = response.get(k).toString();
				if (line.startsWith("event")) {
					String[] parse = line.split(" ");
					parse = parse[2].split("/");
					group = Integer.valueOf(parse[4]);
				} else if (line.startsWith("type")) {
					type = Integer.valueOf(line.substring(line.indexOf(':')+1));
					creatures.add(type);
				} else if (line.startsWith("pos")) {
					int position = Integer.parseInt(line.substring(line.indexOf(':')+1));					
					if (1 == type.intValue()) {
						Creature creature =
							new Creature(null, null, 0, 0, group.intValue(), position);
						creatures.add(creature);
					} else {
						Creature creature =	readCreature(group.intValue(), position);
						creatures.add(creature);						
					}
				}
			}					  
		}		
		return creatures.size() > 0;
	}
	/**
	 * Read simulation messages.
	 * @return true if any messages read
	 * @throws IOException Thrown on I/O error
	 * @throws CommunicationErrorException Thrown on protocol error
	 */
	public boolean getMessages()
		throws IOException, CommunicationErrorException {
		String line = null;
		String request = null;
		ArrayList response = null;
		String buffer = null;

		request = "call " + messageEvent + " sendall";
		response = sendRequest(request);

		//get messages from response
		for (int k = 0; k < response.size(); ++k) {
			line = response.get(k).toString();
			if (line.startsWith("class")) {
				buffer = line.substring(line.indexOf(':')+1) + "::";
			} else if (line.startsWith("function")) {
				buffer += line.substring(line.indexOf(':')+1) + " - ";
			} else if (line.startsWith("message")) {
				messages.add(buffer + line.substring(line.indexOf(':')+1));
			} else if (line.startsWith("level")) {
				messages.add(Integer.valueOf(line.substring(line.indexOf(':')+1)));
			}
		}

		return messages.size() > 0;
	}
	/**
	 * Read running state.
	 * @return true if running state has changed
	 * @throws IOException Thrown on I/O error
	 * @throws CommunicationErrorException Thrown on protocol error
	 */
	public boolean getChangedRunningState()
	throws IOException, CommunicationErrorException {
		String line = null;
		String request = null;
		ArrayList response = null;
		String value = null;

		request = "call " + runningEvent + " sendall";
		response = sendRequest(request);

		//get running state changes from response
		for (int k = 0; k < response.size(); ++k) {
			line = response.get(k).toString();
			if (line.startsWith("value")) {
				value = line.substring(line.indexOf(':')+1);
				running = value.equals("1");
			}
		}
		return value != null;
	}
	/**
	 * Read changes on groups list.
	 * @return true if any changes read
	 * @throws IOException Thrown on I/O error
	 * @throws CommunicationErrorException Thrown on protocol error
	 */
	public boolean getChangedGroups()
	throws IOException, CommunicationErrorException {
			
		String line = null;
		String request = null;
		ArrayList response = null;
		Integer type = null;
		String name = null;
		
		request = "call " + groupEvent + " sendall";
		response = sendRequest(request);

		//get changes from response
		for (int k = 0; k < response.size(); ++k) {
			line = response.get(k).toString();
			if (line.startsWith("type")) {
				type = Integer.valueOf(line.substring(line.indexOf(':')+1));
				groups.add(type);
			} else if (line.startsWith("pos")) {
				Integer position = Integer.valueOf(line.substring(line.indexOf(':')+1));									
				groups.add(position);
				if (0 == type.intValue()) {			
					name = readGroup(position.intValue());
					registerOnCreaturesChanged(position.intValue(), true);
					creaturesCount.add(position.intValue(), new Integer(0));				
				} else if (1 == type.intValue()) {
					name = null;
					creatureEvents.remove(position.intValue());
					creaturesCount.removeElementAt(position.intValue()); 	
				} else if (2 == type.intValue()) {			
					name = readGroup(position.intValue());
					registerOnCreaturesChanged(position.intValue(), false);
					//@TODO review: just a name change or what?
				}
				groups.add(name);
			}
		}
		return groups.size() > 0;
	}		
	/**
	 * Pass data of updated objects to the tree and GL View.
	 * It takes care of groups and creatures.
	 */
	public void updateTree() {
		updateLog();
		
		//update groups 
		for (int i = 0; i < groups.size(); i+=3) {
			Integer type = (Integer)groups.get(i);
			Integer position = (Integer)groups.get(i+1);
			String name = (String)groups.get(i+2);
			updateGroup(type.intValue(), position.intValue(), name);
		}
		groups.clear();
		
		//update creatures
		for (int i = 0; i < creatures.size(); i+=2) {
			Integer type = (Integer)creatures.get(i);
			Creature creature = (Creature)creatures.get(i+1);
			updateCreature(type.intValue(), creature);			
		}
		creatures.clear();
		passObjectsToView();
	}
	/**
	 * Update group data in a tree.
	 * @param type update type (0-add, 1-remove, 2-modify)
	 * @param position group position on population list
	 * @param name a name of a group
	 */
	private void updateGroup(int type, int position, String name) {
		if (0 == type) {								
			DefaultMutableTreeNode node = new DefaultMutableTreeNode(name);
				
			//add [empty] tag if group is empty
			Integer n = (Integer)creaturesCount.get(position);
			if (0 == n.intValue()) {				
				node.add(new DefaultMutableTreeNode("[empty]"));
			}
			getOptions().addTreeNode(node, position);
		} else if (1 == type) { 
			getOptions().removeTreeNode(position);
		} else if (2 == type) {
			updateGroup(1, position, name);
			updateGroup(0, position, name);	
		}
	}
	/**
	 * Update creatures data in a group.
	 * @param type update type (0-add, 1-remove, 2-modify)
	 * @param creature object
	 */
	private void updateCreature(int type, Creature creature) {
		if (0 == type) {				
			//remove [empty] tag if group is empty
			Integer n = (Integer)creaturesCount.get(creature.getGroup());
			if (0 == n.intValue()) {
				getOptions().removeTreeNode(creature.getGroup(), 0);
				creaturesCount.set(creature.getGroup(), new Integer(1));
			}
			DefaultMutableTreeNode node = new DefaultMutableTreeNode(creature);
			getOptions().addTreeNode(node, creature.getGroup(), creature.getIndex());
		} else if (1 == type) { 
			getOptions().removeTreeNode(creature.getGroup(), creature.getIndex());
		} else if (2 == type) {
			updateCreature(1, creature);
			updateCreature(0, creature);
		}	
	}
	/**
	 * Show received messages in log window.
	 */
	public void updateMessages() {
		for (int i = 0; i < messages.size(); i+=2) {
			String line = messages.get(i).toString();
			int type = ((Integer)messages.get(i+1)).intValue();

			Color color = Color.black;
			if (0 == type) { //info
				color = new Color(0, 128, 0); //green
			} else if (1 == type) { //warning
				color = Color.orange;
			} else if (2 == type) { //error
				color = Color.red;
			} else if (3 == type) { //critical error
				color = new Color(237, 164, 61); //crimson
			}
			networkLogs.addLogLine(line, color);
		}
		messages.clear();
	}
	/**
	 * Activate/disactivate buttons. Add message to status bar.
	 */	
	public void updateRunningState() {
		updateLog();
		if (running) {
			options.setStarted(true);
			options.setStopped(false);
			statusBar.addMessage("Simulation is running.", StatusBar.DEFAULT);
		} else {
			options.setStarted(false);
			options.setStopped(true);
			statusBar.addMessage("Simulation stopped.", StatusBar.DEFAULT);
		}		
	}
}
