/* 
 *(C) Poznan University of Technology
 * File name: ViewGL.java.
 * Created on 2004-01-07 00:35:38.
 */
package frams_client_3d;

import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.util.ArrayList;

import javax.swing.JPanel;

import jgl.GL;
import jgl.GLU;
import jgl.GLUT;

/**
 * GLWindow class.
 * @author <a href="mailto:momat@man.poznan.pl">MoMaT</a>
 */
public class ViewGL extends JPanel { 

	private static float ZOOM_SCALE = 0.05f;
	private static float MOVE_SCALE = 0.02f;
	private static float ROTATE_SCALE = 0.01f;
	private static int WHEEL_SCALE = 25;
	
	//initial camera position
	private static float eye[] = {10, 10, 30};
	private static float view[] = {10, 10, 0};
	private static float up[] = {0, 1, 0};
	private static Vector3D initialViewVector =
		new Vector3D(view[0]-eye[0], view[1]-eye[1], view[2]-eye[2]);		 

	private Camera camera = new Camera();
	private boolean ready = false;
	private int displayList;
	private int heighfield;
	private int creaturesCount; 	
	
	private boolean isLeftButtonPressed = false;
	private boolean isRightButtonPressed = false;
	private boolean isMiddleButtonPressed = false;
	private int lastX;
	private int lastY;

	GL myGL = new GL();
	GLU myGLU = new GLU(myGL);
	GLUT myUT = new GLUT(myGL);

	/**
	 * Initialize GL window and set event handlers functions.
	 */
	public void init() {

		myUT.glutInitWindowSize(500, 400);
		myUT.glutInitWindowPosition(0, 0);
		myUT.glutCreateWindow(this);
		initGL();
		myUT.glutDisplayFunc("display");		
		
		initialViewVector.normalize();
		
		this.addMouseListener(new java.awt.event.MouseAdapter() { 
        	public void mousePressed(java.awt.event.MouseEvent e) {            		
        		mouse(e.getButton(), true, e.getX(), e.getY());
        	}
			public void mouseReleased(java.awt.event.MouseEvent e) {            		
				mouse(e.getButton(), false, e.getX(), e.getY());
			}
        });
        this.addMouseMotionListener(new java.awt.event.MouseMotionAdapter() { 
        	public void mouseDragged(java.awt.event.MouseEvent e) {    
        		motion(e.getX(), e.getY());
        	}
        });
		this.addMouseWheelListener(new java.awt.event.MouseWheelListener() { 
			public void mouseWheelMoved(java.awt.event.MouseWheelEvent e) {
				isMiddleButtonPressed = true;
				motion(0, lastY+e.getWheelRotation()*WHEEL_SCALE);
				isMiddleButtonPressed = false;			
			}
		});
        this.addComponentListener(new java.awt.event.ComponentAdapter() { 
        	public void componentResized(java.awt.event.ComponentEvent e) {    
        		reshapeWindow(getSize().width, getSize().height);
        	}
        });
		
		myUT.glutMainLoop();
	}
	/**
	 * @see java.awt.Component#update(java.awt.Graphics)
	 */
	public void update (Graphics g) {
		paint (g);
	}
	/**
	 * @see java.awt.Component#paint(java.awt.Graphics)
	 */
	public void paint (Graphics g) {
		myGL.glXSwapBuffers (g, this);
	}
	/**
	 * Mouse button pressed event handler.
	 * @param button button type
	 * @param isDown true if button is pressed 
	 * @param x
	 * @param y
	 */
	public void mouse(int button, boolean isDown, int x, int y) {
		if (button == MouseEvent.BUTTON1) {
			if (isDown) {
				isLeftButtonPressed = true;
				lastX = x;
				lastY = y;
			} else {
				isLeftButtonPressed = false;
			}
		} else if (button == MouseEvent.BUTTON3) {
			if (isDown) {
				isRightButtonPressed = true;
				lastX = x;
				lastY = y;
			} else {
				isRightButtonPressed = false;
			}
		} else if (button == MouseEvent.BUTTON2) {
			if (isDown) {
				isMiddleButtonPressed = true;
				lastY = y;
			} else {
				isMiddleButtonPressed = false;
			}
		}
	}
	/**
	 * Mouse moved event handler.
	 * @param x
	 * @param y
	 */
	public void motion(int x, int y) {
		if (isLeftButtonPressed) {
			float angleY = (float)((lastX-x)*ROTATE_SCALE);        			        
			float angleZ = (float)((y-lastY)*ROTATE_SCALE);
												
			camera.rotateVertical(angleZ);
			camera.rotateHorizontal(angleY);
			Vector3D viewVector = camera.getViewVector();
			float cos = viewVector.scalarProduct(initialViewVector)/viewVector.length();
			if (cos > 0) {
				myUT.glutPostRedisplay();
			} else {
				camera.rotateVertical(-angleZ);
				camera.rotateHorizontal(-angleY);
			}
		} else if (isRightButtonPressed) {
			float stepX = (float)((lastX-x)*MOVE_SCALE);        			        
			float stepY = (float)((y-lastY)*MOVE_SCALE);

			camera.strafe(stepX);
			camera.level(stepY);
			myUT.glutPostRedisplay();
		} else if (isMiddleButtonPressed) {      			        
			float stepY = (float)((y-lastY)*ZOOM_SCALE);

			camera.zoom(-stepY);
			myUT.glutPostRedisplay();
		}
		lastX = x;
		lastY = y;		
	}	
	/**
	 * Change window shape if GL panel is resized.
	 * @param w new window width
	 * @param h new window height
	 */
	public void reshapeWindow(int w, int h) {
		myGL.glViewport(0, 0, w, h);
		myGL.glMatrixMode(GL.GL_PROJECTION);
		myGL.glLoadIdentity();
		myGLU.gluPerspective(45.0,(double)w/(double)h, 0.1, 200.0);
		myGL.glMatrixMode(GL.GL_MODELVIEW);
		myGL.glLoadIdentity();
		myUT.glutPostRedisplay();
	}	
	/**
	 * GL initial settings.
	 */
	private void initGL() {
		camera.setPosition(eye, view, up);
		myGLU.gluLookAt(eye[0], eye[1], eye[2], view[0], view[1], view[2], up[0], up[1], up[2]);		
		
		myGL.glShadeModel(GL.GL_FLAT);
				
		myGL.glClearDepth(1.0f);
		myGL.glDepthFunc(GL.GL_LEQUAL);
		myGL.glEnable(GL.GL_DEPTH_TEST);

/*
		float LightAmbient[]= {0.5f, 0.5f, 0.5f, 1.0f};
		float LightDiffuse[]= {1.0f, 1.0f, 1.0f, 1.0f};		

		myGL.glLightfv(GL.GL_LIGHT1, GL.GL_AMBIENT, LightAmbient);
		myGL.glLightfv(GL.GL_LIGHT1, GL.GL_DIFFUSE, LightDiffuse);
		
		myGL.glEnable(GL.GL_LIGHT1);
		myGL.glEnable(GL.GL_LIGHTING);		
*/
		myGL.glEnable(GL.GL_COLOR_MATERIAL);		
	}
	/**
	 * GL render view.
	 */
	public void display() {				
		float eye[] = camera.getEye();
		float view[] = camera.getView();
		float up[] = camera.getUp();
		//float LightPosition[]= {10.0f, 10.0f, 10.0f, 0};				
		
		myGL.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);				
		myGL.glLoadIdentity();				
		myGLU.gluLookAt(eye[0], eye[1], eye[2], view[0], view[1], view[2], up[0], up[1], up[2]);		
		//myGL.glLightfv(GL.GL_LIGHT1, GL.GL_POSITION, LightPosition);
						
		if (heighfield != 0) {
			myGL.glCallList(heighfield);
		}
		for (int i = 0; i < creaturesCount; ++i) {
			myGL.glCallList(displayList+i);
		}										 		
		myGL.glFlush();				
	}
	/**
	 * Draw a cube translated, rotated and scaled by given vectors.
	 * @param translate translation vector
	 * @param scale scale vector
	 * @param rotate rotation vector
	 * @param color cube color index (0-red, 1-blue)
	 */
	private void drawCube(float translate[], float scale[], float[] rotate, int color) {		
			
		float Diffuse[][] = {{1.0f, 0.0f, 0.0f, 0.5f}, {0.0f, 0.5f, 1.0f, 0.5f}};
		//float Ambient[] = {0.6f, 0.6f, 0.6f, 0.6f}; 
		//float Specular[] = {0.8f, 0.8f, 0.8f, 0.8f};					
		//float Shininess = 60.0f;		

		//myGL.glMaterialfv(GL.GL_FRONT, GL.GL_DIFFUSE, Diffuse[color]);
		//myGL.glMaterialfv(GL.GL_FRONT, GL.GL_AMBIENT, Ambient);
		//myGL.glMaterialfv(GL.GL_FRONT, GL.GL_SPECULAR, Specular);
		//myGL.glMaterialf(GL.GL_FRONT, GL.GL_SHININESS, Shininess);			

		myGL.glColor3fv(Diffuse[color]);
				
		myGL.glTranslatef(translate[0], translate[1], translate[2]);		
		myGL.glRotatef(rotate[0], rotate[1], rotate[2], rotate[3]);
		myGL.glScalef(scale[0], scale[1], scale[2]);		
		myGL.glBegin(GL.GL_QUADS);
			// Front
			myGL.glNormal3f( 0.0f, 0.0f, 1.0f);
			myGL.glVertex3f(-1.0f, -1.0f,  1.0f);
			myGL.glNormal3f( 0.0f, 0.0f, 1.0f);
			myGL.glVertex3f( 1.0f, -1.0f,  1.0f);
			myGL.glNormal3f( 0.0f, 0.0f, 1.0f);
			myGL.glVertex3f( 1.0f,  1.0f,  1.0f);
			myGL.glNormal3f( 0.0f, 0.0f, 1.0f);
			myGL.glVertex3f(-1.0f,  1.0f,  1.0f);
			// Back
			myGL.glNormal3f( 0.0f, 0.0f, -1.0f);
			myGL.glVertex3f(-1.0f, -1.0f, -1.0f);
			myGL.glNormal3f( 0.0f, 0.0f, 1.0f);
			myGL.glVertex3f(-1.0f,  1.0f, -1.0f);
			myGL.glNormal3f( 0.0f, 0.0f, 1.0f);				
			myGL.glVertex3f( 1.0f,  1.0f, -1.0f);
			myGL.glNormal3f( 0.0f, 0.0f, 1.0f);				
			myGL.glVertex3f( 1.0f, -1.0f, -1.0f);
			// Top
			myGL.glNormal3f( 0.0f, 1.0f, 0.0f);
			myGL.glVertex3f(-1.0f,  1.0f, -1.0f);
			myGL.glNormal3f( 0.0f, 1.0f, 0.0f);
			myGL.glVertex3f(-1.0f,  1.0f,  1.0f);
			myGL.glNormal3f( 0.0f, 1.0f, 0.0f);				
			myGL.glVertex3f( 1.0f,  1.0f,  1.0f);
			myGL.glNormal3f( 0.0f, 1.0f, 0.0f);				
			myGL.glVertex3f( 1.0f,  1.0f, -1.0f);
			// Bottom
			myGL.glNormal3f( 0.0f, -1.0f, 0.0f);
			myGL.glVertex3f(-1.0f, -1.0f, -1.0f);
			myGL.glNormal3f( 0.0f, -1.0f, 0.0f);
			myGL.glVertex3f( 1.0f, -1.0f, -1.0f);
			myGL.glNormal3f( 0.0f, -1.0f, 0.0f);				
			myGL.glVertex3f( 1.0f, -1.0f,  1.0f);
			myGL.glNormal3f( 0.0f, -1.0f, 0.0f);				
			myGL.glVertex3f(-1.0f, -1.0f,  1.0f);
			// Right
			myGL.glNormal3f( 1.0f, 0.0f, 0.0f);
			myGL.glNormal3f( 1.0f, 0.0f, 0.0f);
			myGL.glVertex3f( 1.0f, -1.0f, -1.0f);
			myGL.glNormal3f( 1.0f, 0.0f, 0.0f);				
			myGL.glVertex3f( 1.0f,  1.0f, -1.0f);
			myGL.glNormal3f( 1.0f, 0.0f, 0.0f);				
			myGL.glVertex3f( 1.0f,  1.0f,  1.0f);
			myGL.glNormal3f( 1.0f, 0.0f, 0.0f);				
			myGL.glVertex3f( 1.0f, -1.0f,  1.0f);
			// Left
			myGL.glNormal3f( -1.0f, 0.0f, 0.0f);
			myGL.glNormal3f( -1.0f, 0.0f, 0.0f);
			myGL.glVertex3f(-1.0f, -1.0f, -1.0f);
			myGL.glNormal3f( -1.0f, 0.0f, 0.0f);				
			myGL.glVertex3f(-1.0f, -1.0f,  1.0f);
			myGL.glNormal3f( -1.0f, 0.0f, 0.0f);				
			myGL.glVertex3f(-1.0f,  1.0f,  1.0f);
			myGL.glNormal3f( -1.0f, 0.0f, 0.0f);				
			myGL.glVertex3f(-1.0f,  1.0f, -1.0f);
		myGL.glEnd();		
	}	
	/**
	 * Prepare display lists for given creatures.  
	 * @param creatures array of creatures
	 */
	public void setCreatures(ArrayList creatures) {
		creaturesCount = creatures.size();
		displayList = myGL.glGenLists(creaturesCount);
		for (int i = 0; i < creatures.size(); ++i) {
			setCreature((Creature)creatures.get(i), displayList+i);
		}
		myUT.glutPostRedisplay();						
	}
	/**
	 * Prepare display list for a creature.
	 * @param creature a creature object
	 * @param list a display list
	 */
	private void setCreature(Creature creature, int list) {		
		float partScale[] = {0.1f, 0.1f, 0.1f};
		float jointScale[] = {0.05f, 0.05f, 0.05f};
		float rotate[] = {0, 0, 0, 0};
		float r[] = {1, 1, 1};

		float parts[][] = creature.getParts();
		int joints[][] = creature.getJoints();
		
		myGL.glNewList(list, GL.GL_COMPILE);
		for (int i = 0; i < parts.length; ++i) {
			myGL.glPushMatrix();
			drawCube(parts[i], partScale, rotate, 0);
			myGL.glPopMatrix();			
		}
		for (int i = 0; i < joints.length; ++i) {
			jointScale[0] = creature.jointLength(i)/2;
			myGL.glPushMatrix();
			drawCube(creature.jointTranslation(i), jointScale,
				creature.jointRotation(i), 1);
			myGL.glPopMatrix();				 
		}
		myGL.glEndList();		
	}	
	/**
	 * Prepare display list for heighfield.
	 * @param points array of points
	 * @param faces array of faces
	 */
	public void setHeighfield(ArrayList points, ArrayList faces) {
		float Diffuse[] = {1.0f, 1.0f, 0.6f, 1.0f};
		//float Ambient[] = {0.6f, 0.6f, 0.6f, 0.6f}; 
		//float Specular[] = {0, 0, 0, 1};					
		//float Shininess = 0;							

		heighfield = myGL.glGenLists(1);		
		myGL.glNewList(heighfield, GL.GL_COMPILE);		
			//myGL.glMaterialfv(GL.GL_FRONT, GL.GL_DIFFUSE, Diffuse);
			//myGL.glMaterialfv(GL.GL_FRONT, GL.GL_AMBIENT, Ambient);
			//myGL.glMaterialfv(GL.GL_FRONT, GL.GL_SPECULAR, Specular);
			//myGL.glMaterialf(GL.GL_FRONT, GL.GL_SHININESS, Shininess);			

			myGL.glColor3fv(Diffuse);			
									
			for (int i = 0; i < faces.size(); ++i) {
				String face[] = (String[])faces.get(i);
				
				//choose primitive according to number of points
				if (3 == face.length) {
					myGL.glBegin(GL.GL_TRIANGLES);	
				} else if (4 == face.length) {
					myGL.glBegin(GL.GL_QUADS);
				} else {
					myGL.glBegin(GL.GL_POLYGON);
				}
				
				//draw face points
				for (int j = 0; j < face.length; ++j) {
					int index = Integer.parseInt(face[j]);
					String point[] = (String[])points.get(index);
					myGL.glVertex3f(Float.parseFloat(point[0]),
						Float.parseFloat(point[1]), Float.parseFloat(point[2])-0.1f);	
				}				
				myGL.glEnd();
			}
		myGL.glEndList();
		myUT.glutPostRedisplay();
	}
}
